/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.simulator.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import org.gecko.core.tests.AbstractOSGiTest;
import org.gecko.core.tests.ServiceChecker;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.FrameworkUtil;

import de.dim.trafficos.model.device.ConflictingLane;
import de.dim.trafficos.model.device.Intersection;
import de.dim.trafficos.model.device.Phase;
import de.dim.trafficos.model.device.PhaseGroup;
import de.dim.trafficos.model.device.Program;
import de.dim.trafficos.model.device.ScheduleModeType;
import de.dim.trafficos.simulator.api.IntersectionConstants;
import de.dim.trafficos.simulator.api.IntersectionService;
import de.dim.trafficos.simulator.api.SignalPlanConstants;
import de.dim.trafficos.simulator.api.SignalPlanService;


/**
 * Tests the implementation of the SignalPlanService when creating a Program for a 4 arms intersection
 * @since 1.0
 */
@RunWith(MockitoJUnitRunner.class)
public class SigPlanProgram4ArmsTest extends AbstractOSGiTest {
	
	
	private IntersectionService service;
	private SignalPlanService sigService;
	private Map<Integer, String> options;
	private File file = new File("data/TestFixProgram4Arms.txt");
	
	/**
	 * Creates a new instance.
	 * @param bundleContext
	 */
	public SigPlanProgram4ArmsTest() {
		super(FrameworkUtil.getBundle(SigPlanProgram4ArmsTest.class).getBundleContext());
	}

	/**
	 * Here you can put everything you want to be executed before every test
	 */
	public void doBefore() {
		if(file.exists()) {
			file.delete();
		}
		options = new HashMap<Integer, String>();
	}
	
	/**
	 * Here you can put everything you want to be executed after every test
	 */
	public void doAfter() {
		
	}
	
	private void setupServices() {
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		service = getService(IntersectionService.class);
		assertNotNull(service);
		
		ServiceChecker<SignalPlanService> sigChecker = createCheckerTrackedForCleanUp(SignalPlanService.class);
		sigChecker.start();
		assertNotNull(sigChecker);
		assertEquals(1, sigChecker.getCurrentCreateCount(true));
		
		sigService = getService(SignalPlanService.class);
		assertNotNull(sigService);	
	}
	
	/**
	 * This test would create a table in a file with the status of the SignalGroup of each IncomingLane
	 * for each value of TX. 
	 * 
	 * @throws IOException
	 */
	@Test
	public void test4ArmsSigProgramCreation() throws IOException {
		setupServices();
		
		options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		assertFalse(phases.isEmpty());
		
		sigService.createPhaseGroups(intersection);
		assertFalse(intersection.getPhaseGroup().isEmpty());
		List<PhaseGroup> phGroups = intersection.getPhaseGroup().stream().sorted(Comparator.comparing(PhaseGroup::getPenalty))
				.collect(Collectors.toList());
		PhaseGroup pg = phGroups.get(2);
		List<Phase> groupPh = pg.getPhase();
		for(Phase p : groupPh) {			
			if(p.getId().equals("PH_13")) {
				p.setWeightMax(70);
				p.setWeightMin(35);
			}
			else if(p.getId().equals("PH_15")) {
				p.setWeightMax(50);
				p.setWeightMin(15);
			}
			else if(p.getId().equals("PH_21")) {
				p.setWeightMax(50);
				p.setWeightMin(35);
			}
			else if(p.getId().equals("PH_3")) {
				p.setWeightMax(20);
				p.setWeightMin(15);
			}
		}
		
		Program program = sigService.createFixTimeProgram(pg, UUID.randomUUID().toString(), 100);
		Map<Integer, Map<ConflictingLane, String>> programMap = sigService.applyProgram(intersection, program, ScheduleModeType.WORKING_DAY);
		assertNotNull(programMap);
		assertFalse(programMap.isEmpty());
		
		file.createNewFile();
		BufferedWriter writer = new BufferedWriter(new FileWriter(file.getAbsolutePath(), true));
		for(int m = 0; m < programMap.size(); m++) {
			if(m == 0) {
				Iterator<ConflictingLane> iter = programMap.get(m).keySet().iterator();
				while(iter.hasNext()) {
					String id = iter.next().getId();
					if(id.startsWith("P")) {
						writer.append("\t" + id);
					}
					else {
						writer.append("\t\t" + id);
					}
					
				}
				writer.append("\n");
			}
			writer.append(String.valueOf(m));
			Map<ConflictingLane, String> values = programMap.get(m);
			values.entrySet().forEach(e->{
				try {
					writer.append("\t\t" + e.getValue());
				} catch (IOException e1) {
					e1.printStackTrace();
				}
			});
			writer.append("\n");	
		}	     
	    writer.close();
	}
}
