/**
 * Copyright (c) 2012 - 2017 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.emf.osgi.bson;

import java.text.SimpleDateFormat;
import java.util.Locale;
import java.util.TimeZone;

import org.eclipse.emf.ecore.resource.Resource.Factory.Registry;
import org.emfjson.jackson.module.EMFModule;
import org.emfjson.jackson.resource.JsonResourceFactory;
import org.gecko.emf.osgi.ResourceFactoryConfigurator;
import org.gecko.emf.osgi.annotation.EMFResourceFactoryConfigurator;
import org.gecko.emf.osgi.annotation.provide.ProvideEMFResourceConfigurator;
import org.gecko.emf.osgi.json.configuration.ConfigurableJsonResourceFactory;
import org.osgi.service.component.annotations.Component;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;

import de.undercouch.bson4jackson.BsonFactory;

/**
 * Implementation of the {@link IResourceSetConfigurator} to provide support for {@link JsResourceImpl}.
 * 
 * It provides the {@link JsResourceFactoryImpl} for the following identifiers:
 * <ul>
 * 	<li>Extension: json
 * 	<li>contentType: application/json
 * 	<li>contentType: application/x-json
 * 	<li>Protocol: json
 * </ul>
 * 
 * @author Juergen Albert
 * @since 27.06.2014
 */
@Component(name="EMFBsonConfigurator", immediate=true, service=ResourceFactoryConfigurator.class)
@ProvideEMFResourceConfigurator(name="EMFBson", 
	contentType = {
			"application/bson"
	},
	fileExtension = {
			"bson"
	},
	version = "1.0.1"
)
@EMFResourceFactoryConfigurator(
		emf_resource_configurator_name ="EMFJson",
		emf_resource_configurator_fileExtension = {
				"bson"
		},
		emf_resource_configurator_contentType = {
				"application/bson"
			}
		)
public class EMFBsonResourceFactoryConfigurator implements ResourceFactoryConfigurator{

	/* 
	 * (non-Javadoc)
	 * @see de.dim.emf.osgi.ResourceFactoryConfigurator#configureResourceFactory(org.eclipse.emf.ecore.resource.Resource.Factory.Registry)
	 */
	@Override
	public void configureResourceFactory(Registry registry) {

		registry.getExtensionToFactoryMap().put("bson", createBsonFactory());
		registry.getContentTypeToFactoryMap().put("application/bson", createBsonFactory());
		registry.getProtocolToFactoryMap().put("bson", createBsonFactory());
	}
	
	private JsonResourceFactory createBsonFactory() {
		ObjectMapper mapper = new ObjectMapper(new BsonFactory());
		final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss", Locale.ENGLISH);
		dateFormat.setTimeZone(TimeZone.getDefault());

		mapper.configure(SerializationFeature.INDENT_OUTPUT, true);
		mapper.setDateFormat(dateFormat);
		mapper.setTimeZone(TimeZone.getDefault());
		mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        mapper.configure(DeserializationFeature.ACCEPT_EMPTY_STRING_AS_NULL_OBJECT, true);
        mapper.configure(DeserializationFeature.ACCEPT_EMPTY_ARRAY_AS_NULL_OBJECT, true);
        mapper.disable(com.fasterxml.jackson.databind.SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        mapper.enable(com.fasterxml.jackson.databind.SerializationFeature.WRITE_DATES_WITH_ZONE_ID);
		mapper.registerModule(new EMFModule());
		
		return new ConfigurableJsonResourceFactory(mapper);
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.emf.osgi.ResourceFactoryConfigurator#unconfigureResourceFactory(org.eclipse.emf.ecore.resource.Resource.Factory.Registry)
	 */
	@Override
	public void unconfigureResourceFactory(Registry registry) {
		registry.getExtensionToFactoryMap().remove("bson");
		registry.getProtocolToFactoryMap().remove("bson");
		registry.getContentTypeToFactoryMap().remove("application/bson");
	}

}
