/**
 * Copyright (c) 2012 - 2017 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.qvt.osgi.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Dictionary;
import java.util.Hashtable;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.emf.osgi.EPackageConfigurator;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.emf.osgi.model.test.GenderType;
import org.gecko.emf.osgi.model.test.Person;
import org.gecko.emf.osgi.model.test.TestFactory;
import org.gecko.emf.osgi.model.test.TestPackage;
import org.gecko.qvt.osgi.api.ModelTransformationConstants;
import org.gecko.qvt.osgi.api.ModelTransformationFactory;
import org.gecko.qvt.osgi.api.ModelTransformator;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Filter;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;

/**
 * Test QVT Mapping
 * @author mark
 * @since 20.10.2017
 */
@RunWith(MockitoJUnitRunner.class)
public class ConfigurableTransformatorModelBindingIntegrationTest extends BasicExampleTest {

	private final BundleContext context = FrameworkUtil.getBundle(ConfigurableTransformatorModelBindingIntegrationTest.class).getBundleContext();
	private ServiceRegistration<EPackageConfigurator> configRegistration = null;

	@Before
	public void before() {
		setup(context);
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put("emf.model.name", "test");
		configRegistration = context.registerService(EPackageConfigurator.class, new TestConfigurator(), props);
	}

	@After
	public void after() {
		configRegistration.unregister();
	}

	@Test
	public void testWithModelBinding() throws InterruptedException, URISyntaxException, IOException, InvalidSyntaxException {
		ResourceSetFactory rsf = getService(ResourceSetFactory.class, 3000);
		assertNotNull(rsf);
		ResourceSet rs = rsf.createResourceSet();
		EPackage ePackage = rs.getPackageRegistry().getEPackage(TestPackage.eNS_URI);
		assertNotNull(ePackage);
		Resource r1 = rs.createResource(URI.createURI("tmp.test"));
		Person p1 = TestFactory.eINSTANCE.createPerson();
		p1.setFirstName("Mark");
		p1.setLastName("Hoffmann");
		p1.setGender(GenderType.MALE);
		r1.getContents().add(p1);
		
		ConfigurationAdmin configAdmin = getService(ConfigurationAdmin.class, 3000);
		assertNotNull(configAdmin);
		
		ModelTransformationFactory mtf = getService(ModelTransformationFactory.class, 3000);
		assertNotNull(mtf);
		
		Filter f = FrameworkUtil.createFilter("(" + ModelTransformationConstants.TRANSFORMATOR_NAME + "=test-trafo)");
		ModelTransformator transformator = getService(f, 2000);
		assertNull(transformator);
		
		Configuration configuration = configAdmin.getFactoryConfiguration(ModelTransformationConstants.CONFIGURABLE_TRANSFORMATOR_FACTORY_ID, "test-trafo", "?");
		assertNotNull(configuration);
		assertNull(configuration.getProperties());
		
		Dictionary<String, Object> properties = new Hashtable<String, Object>();
		properties.put(ModelTransformationConstants.TRANSFORMATOR_NAME, "test-trafo");
		String path = context.getBundle().getSymbolicName();
		path += ":"  + context.getBundle().getVersion().toString();
		path += "/PersonTransformation.qvto";
		properties.put(ModelTransformationConstants.TEMPLATE_PATH, path);
		properties.put(ModelTransformationConstants.MODEL_TARGET, "(emf.model.name=test)");
		
		configuration.update(properties);
		
		transformator = getService(f, 3000);
		assertNotNull(transformator);
		
		EObject result = transformator.startTransformation(p1);
		assertNotNull(result);
		assertTrue(result instanceof Person);
		Person resultPerson = (Person) result;
		assertEquals(GenderType.FEMALE, resultPerson.getGender());
		assertEquals("Markin", resultPerson.getFirstName());
		assertEquals("Hoffmannin", resultPerson.getLastName());
		
		configuration.delete();
		Thread.sleep(1000l);
		transformator = getService(f, 1000);
		assertNull(transformator);
	}
	
	@Test
	public void testWrongModelBinding() throws InterruptedException, URISyntaxException, IOException, InvalidSyntaxException {
		ResourceSetFactory rsf = getService(ResourceSetFactory.class, 3000);
		assertNotNull(rsf);
		ResourceSet rs = rsf.createResourceSet();
		EPackage ePackage = rs.getPackageRegistry().getEPackage(TestPackage.eNS_URI);
		assertNotNull(ePackage);
		Resource r1 = rs.createResource(URI.createURI("tmp.test"));
		Person p1 = TestFactory.eINSTANCE.createPerson();
		p1.setFirstName("Mark");
		p1.setLastName("Hoffmann");
		p1.setGender(GenderType.MALE);
		r1.getContents().add(p1);
		
		ConfigurationAdmin configAdmin = getService(ConfigurationAdmin.class, 3000);
		assertNotNull(configAdmin);
		
		ModelTransformationFactory mtf = getService(ModelTransformationFactory.class, 3000);
		assertNotNull(mtf);
		
		Filter f = FrameworkUtil.createFilter("(" + ModelTransformationConstants.TRANSFORMATOR_NAME + "=test-trafo)");
		ModelTransformator transformator = getService(f, 2000);
		assertNull(transformator);
		
		Configuration configuration = configAdmin.getFactoryConfiguration(ModelTransformationConstants.CONFIGURABLE_TRANSFORMATOR_FACTORY_ID, "test-trafo", "?");
		assertNotNull(configuration);
		assertNull(configuration.getProperties());
		
		Dictionary<String, Object> properties = new Hashtable<String, Object>();
		properties.put(ModelTransformationConstants.TRANSFORMATOR_NAME, "test-trafo");
		String path = context.getBundle().getSymbolicName();
		path += ":"  + context.getBundle().getVersion().toString();
		path += "/PersonTransformation.qvto";
		properties.put(ModelTransformationConstants.TEMPLATE_PATH, path);
		properties.put(ModelTransformationConstants.MODEL_TARGET, "(emf.model.name=test-somthing)");
		
		configuration.update(properties);
		
		transformator = getService(f, 3000);
		assertNull(transformator);
		
	}
	
}