/**
 * Copyright (c) 2012 - 2017 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.emf.repository.mongo.tests;


import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.emf.osgi.EPackageConfigurator;
import org.gecko.emf.osgi.ResourceFactoryConfigurator;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.emf.osgi.model.test.Address;
import org.gecko.emf.osgi.model.test.Person;
import org.gecko.emf.osgi.model.test.TestFactory;
import org.gecko.emf.osgi.model.test.TestPackage;
import org.gecko.emf.osgi.model.test.configurator.TestPackageConfigurator;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.mongo.osgi.MongoClientProvider;
import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.gecko.mongo.osgi.configuration.ConfigurationProperties;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.osgi.framework.Filter;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.util.tracker.ServiceTracker;

import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;

/**
 * Integration tests for the complete EMF mongo repository setup
 * @author Mark Hoffmann
 * @since 26.07.2017
 */
@RunWith(MockitoJUnitRunner.class)
public class MongoRepositoryIntegrationTest {

	private final BundleContext context = FrameworkUtil.getBundle(MongoRepositoryIntegrationTest.class).getBundleContext();
	private MongoClient client;
	private MongoCollection<?> collection;
	private String mongoHost = System.getProperty("mongo.host", "localhost");
	private ServiceRegistration<?> testPackageRegistration = null;

	@Before
	public void setup() throws BundleException {
		MongoClientOptions options = MongoClientOptions.builder().build();
		client = new MongoClient(mongoHost, options);
		testPackageRegistration = context.registerService(new String[] {EPackageConfigurator.class.getName(), ResourceFactoryConfigurator.class.getName()}, new TestPackageConfigurator(), null);
	}

	@After
	public void tearDown() throws InterruptedException {
		if (collection != null) {
			collection.drop();
		}
		if (client != null) {
			client.close();
		}
		if (testPackageRegistration != null) {
			testPackageRegistration.unregister();
			testPackageRegistration = null;
		}
	}

	@Test
	public void testEMFMongoRepository() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {

		// create mongo client configuration
		ServiceReference<ConfigurationAdmin> caRef = context.getServiceReference(ConfigurationAdmin.class);
		assertNotNull(caRef);
		ConfigurationAdmin cm = context.getService(caRef);
		assertNotNull(cm);
		Configuration clientConfig = cm.getConfiguration(ConfigurationProperties.CLIENT_PID, "?");
		assertNotNull(clientConfig);

		// has to be a new configuration
		Dictionary<String, Object> p = clientConfig.getProperties();
		assertNull(p);

		// add service properties
		String clientId = "testClient";
		String clientUri = "mongodb://" + mongoHost + ":27017";
		p = new Hashtable<String, Object>();
		p.put(MongoClientProvider.PROP_CLIENT_ID, clientId);
		p.put(MongoClientProvider.PROP_URI, clientUri);
		clientConfig.update(p);

		MongoClientProvider clientProvider = getService(MongoClientProvider.class, 5000);
		assertNotNull(clientProvider);

		// create data base provider configuration
		Configuration databaseConfig = cm.getConfiguration(ConfigurationProperties.DATABASE_PID, "?");
		assertNotNull(databaseConfig);

		// add service properties
		String dbAlias = "testDB";
		String db = "test";
		Dictionary<String, Object> dbp = new Hashtable<String, Object>();
		dbp.put(MongoDatabaseProvider.PROP_ALIAS, dbAlias);
		dbp.put(MongoDatabaseProvider.PROP_DATABASE, db);
		databaseConfig.update(dbp);
		// now track the MongoDatabaseProvider service

		MongoDatabaseProvider dbProvider = getService(MongoDatabaseProvider.class, 5000);
		assertNotNull(dbProvider);

		ResourceSetFactory rsf = getService(ResourceSetFactory.class, 5000l);
		assertNotNull(rsf);
		
		Configuration config = cm.createFactoryConfiguration("mongoRepository", "?");
		assertNotNull(config);
		Dictionary<String, Object> properties = new Hashtable<>();

		String repoId = "test_repo";
		properties.put(EMFRepository.PROP_ID, repoId);
		properties.put(EMFRepository.PROP_BASE_URI, clientUri + "/" + db);
		config.update(properties);
		
		MongoDatabase database = client.getDatabase(db);
		collection = database.getCollection("Person"); 
		collection.drop();
		
		CountDownLatch latch = new CountDownLatch(1);
		latch.await(1, TimeUnit.SECONDS);
		
		EMFRepository repository = getService(5000l, "(" + EMFRepository.PROP_ID + "=" + repoId + ")");
		assertNotNull(repository);

		Person person = TestFactory.eINSTANCE.createPerson();
		person.setId("test");
		person.setFirstName("Emil");
		person.setLastName("Tester");
		URI uri = repository.createUri(person);
		assertEquals(clientUri + "/" + db + "/Person/test", uri.toString());
		assertEquals(0, collection.count());
		
		repository.save(person);
		
		assertEquals(1, collection.count());
		
		Resource r = person.eResource();
		assertNotNull(r);
		ResourceSet rs = r.getResourceSet();
		assertNotNull(rs);
		assertEquals(1, rs.getResources().size());

		repository.detach(person);
		assertNull(person.eResource());
		assertEquals(0, rs.getResources().size());

		Person personResult = repository.getEObject(TestPackage.Literals.PERSON, "test");
		assertNotNull(personResult);
		assertNotEquals(person, personResult);
		assertNotEquals(r, personResult.eResource());

		assertTrue(EcoreUtil.equals(person, personResult));

		config.delete();
		databaseConfig.delete();
		clientConfig.delete();
	}

	@Test
	public void testEMFMongoRepositorySaveMultiple() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {
		
		// create mongo client configuration
		ServiceReference<ConfigurationAdmin> caRef = context.getServiceReference(ConfigurationAdmin.class);
		assertNotNull(caRef);
		ConfigurationAdmin cm = context.getService(caRef);
		assertNotNull(cm);
		Configuration clientConfig = cm.getConfiguration(ConfigurationProperties.CLIENT_PID, "?");
		assertNotNull(clientConfig);
		
		// has to be a new configuration
		Dictionary<String, Object> p = clientConfig.getProperties();
		assertNull(p);
		
		// add service properties
		String clientId = "testClient";
		String clientUri = "mongodb://" + mongoHost + ":27017";
		p = new Hashtable<String, Object>();
		p.put(MongoClientProvider.PROP_CLIENT_ID, clientId);
		p.put(MongoClientProvider.PROP_URI, clientUri);
		clientConfig.update(p);
		
		MongoClientProvider clientProvider = getService(MongoClientProvider.class, 5000);
		assertNotNull(clientProvider);
		
		// create data base provider configuration
		Configuration databaseConfig = cm.getConfiguration(ConfigurationProperties.DATABASE_PID, "?");
		assertNotNull(databaseConfig);
		
		// add service properties
		String dbAlias = "testDB";
		String db = "test";
		Dictionary<String, Object> dbp = new Hashtable<String, Object>();
		dbp.put(MongoDatabaseProvider.PROP_ALIAS, dbAlias);
		dbp.put(MongoDatabaseProvider.PROP_DATABASE, db);
		databaseConfig.update(dbp);
		// now track the MongoDatabaseProvider service
		
		MongoDatabaseProvider dbProvider = getService(MongoDatabaseProvider.class, 5000);
		assertNotNull(dbProvider);
		
		ResourceSetFactory rsf = getService(ResourceSetFactory.class, 5000l);
		assertNotNull(rsf);
		
		Configuration config = cm.createFactoryConfiguration("mongoRepository", "?");
		assertNotNull(config);
		Dictionary<String, Object> properties = new Hashtable<>();
		
		String repoId = "test_repo";
		properties.put(EMFRepository.PROP_ID, repoId);
		properties.put(EMFRepository.PROP_BASE_URI, clientUri + "/" + db);
		config.update(properties);
		
		MongoDatabase database = client.getDatabase(db);
		collection = database.getCollection("Person"); 
		collection.drop();
		
		CountDownLatch latch = new CountDownLatch(1);
		latch.await(1, TimeUnit.SECONDS);
		
		EMFRepository repository = getService(5000l, "(" + EMFRepository.PROP_ID + "=" + repoId + ")");
		assertNotNull(repository);
		
		Address address = TestFactory.eINSTANCE.createAddress();
		address.setId("testAddress");
		
		Person person = TestFactory.eINSTANCE.createPerson();
		person.setId("test");
		person.setFirstName("Emil");
		person.setLastName("Tester");
		person.setAddress(address);
		URI uri = repository.createUri(person);
		assertEquals(clientUri + "/" + db + "/Person/test", uri.toString());
		assertEquals(0, collection.count());
		
		repository.save(person);
		
		assertEquals(1, collection.count());
		
		Resource r = person.eResource();
		assertNotNull(r);
		ResourceSet rs = r.getResourceSet();
		assertNotNull(rs);
		assertEquals(1, rs.getResources().size());
		
		repository.detach(person);
		assertNull(person.eResource());
		assertEquals(0, rs.getResources().size());
		
		Person personResult = repository.getEObject(TestPackage.Literals.PERSON, "test");
		assertNotNull(personResult);
		assertNotEquals(person, personResult);
		assertNotEquals(r, personResult.eResource());
		
		assertFalse(EcoreUtil.equals(person, personResult));
		assertNotNull(personResult.getAddress());
		assertTrue(personResult.getAddress().eIsProxy());
		
		repository.detach(personResult);
		
		repository.save(person, address);
		
		repository.detach(person);
		repository.detach(address);
		
		personResult = repository.getEObject(TestPackage.Literals.PERSON, "test");
		assertNotNull(personResult);
		assertNotEquals(person, personResult);
		assertNotEquals(r, personResult.eResource());
		
		assertTrue(EcoreUtil.equals(person, personResult));
		assertNotNull(personResult.getAddress());
		assertFalse(personResult.getAddress().eIsProxy());
		
		assertNotEquals(address, personResult.getAddress());
		
		config.delete();
		databaseConfig.delete();
		clientConfig.delete();
	}
	
	<T> T getService(long timeout, String filter) throws InterruptedException, InvalidSyntaxException {
		Filter f = FrameworkUtil.createFilter(filter);
		ServiceTracker<T, T> tracker = new ServiceTracker<>(context, f, null);
		tracker.open();
		return tracker.waitForService(timeout);
	}

	<T> T getService(Class<T> clazz, long timeout) throws InterruptedException, InvalidSyntaxException {
		ServiceTracker<T, T> tracker = new ServiceTracker<>(context, clazz, null);
		tracker.open();
		return tracker.waitForService(timeout);
	}

}
