/**
 * Copyright (c) 2012 - 2017 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.emf.repository.mongo.tests;


import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.emf.osgi.EPackageConfigurator;
import org.gecko.emf.osgi.ResourceFactoryConfigurator;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.emf.osgi.model.test.Person;
import org.gecko.emf.osgi.model.test.TestFactory;
import org.gecko.emf.osgi.model.test.TestPackage;
import org.gecko.emf.osgi.model.test.configurator.TestPackageConfigurator;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.mongo.annotations.RequireMongoEMFRepository;
import org.gecko.emf.repository.mongo.api.EMFMongoConfiguratorConstants;
import org.gecko.mongo.osgi.MongoClientProvider;
import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.osgi.framework.Filter;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceObjects;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.util.tracker.ServiceTracker;

import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;

/**
 * Integration tests for the complete EMF mongo repository setup
 * @author Mark Hoffmann
 * @since 26.07.2017
 */
@RequireMongoEMFRepository
@RunWith(MockitoJUnitRunner.class)
public class MongoConfiguratorIntegrationTest {

	private final BundleContext context = FrameworkUtil.getBundle(MongoConfiguratorIntegrationTest.class).getBundleContext();
	private MongoClient client;
	private MongoCollection<?> collection;
	private String mongoHost = System.getProperty("mongo.host", "localhost");
	private ServiceRegistration<?> testPackageRegistration = null;
	private Configuration repositoryConfig;

	@Before
	public void setup() throws BundleException {
		MongoClientOptions options = MongoClientOptions.builder().build();
		client = new MongoClient(mongoHost, options);
		testPackageRegistration = context.registerService(new String[] {EPackageConfigurator.class.getName(), ResourceFactoryConfigurator.class.getName()}, new TestPackageConfigurator(), null);
	}

	@After
	public void tearDown() throws InterruptedException, IOException {
		if (collection != null) {
			collection.drop();
		}
		if (client != null) {
			client.close();
		}
		if (testPackageRegistration != null) {
			testPackageRegistration.unregister();
			testPackageRegistration = null;
		}
		
		if(repositoryConfig != null) {
			repositoryConfig.delete();
		}
		
		CountDownLatch latch = new CountDownLatch(1);
		latch.await(3, TimeUnit.SECONDS);
	}

	@Test
	public void testEMFMongoRepository() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {

		// create mongo client configuration
		ServiceReference<ConfigurationAdmin> caRef = context.getServiceReference(ConfigurationAdmin.class);
		assertNotNull(caRef);
		ConfigurationAdmin cm = context.getService(caRef);
		assertNotNull(cm);

		repositoryConfig = cm.createFactoryConfiguration(EMFMongoConfiguratorConstants.EMF_MONGO_REPOSITORY_CONFIGURATOR_CONFIGURATION_NAME, "?");
		assertNotNull(repositoryConfig);

		// has to be a new configuration
		assertNull(repositoryConfig.getProperties());

		/**
		 * mongo.instances=test1
		 * test1.baseUris=mongodb://localhost
		 * test1.databases=test
		 */

		Dictionary<String, Object> configProperties = new Hashtable<>();
		configProperties.put("mongo.instances", "test1");
		configProperties.put("test1.baseUris", "mongodb://" + mongoHost);
		configProperties.put("test1.databases", "test");

		repositoryConfig.update(configProperties);

		String clientId = "test1.test";

		MongoClientProvider clientProvider = getService(MongoClientProvider.class, 5000);
		assertNotNull(clientProvider);

		MongoDatabaseProvider dbProvider = getService(MongoDatabaseProvider.class, 5000);
		assertNotNull(dbProvider);

		ResourceSetFactory rsf = getService(ResourceSetFactory.class, 5000l);
		assertNotNull(rsf);

		EMFRepository repository = getService(5000l, "(" + EMFRepository.PROP_ID + "=" + clientId + ")");
		assertNotNull(repository);

		Person person = TestFactory.eINSTANCE.createPerson();
		person.setId("test");
		person.setFirstName("Emil");
		person.setLastName("Tester");
		URI uri = repository.createUri(person);
		assertEquals("mongodb://test1/test/Person/test", uri.toString());

		MongoDatabase database = client.getDatabase("test");
		collection = database.getCollection("Person"); 
		collection.drop();

		assertEquals(0, collection.count());

		CountDownLatch latch = new CountDownLatch(1);
		latch.await(1, TimeUnit.SECONDS);
		
		repository.save(person);

		assertEquals(1, collection.count());

		Resource r = person.eResource();
		assertNotNull(r);
		ResourceSet rs = r.getResourceSet();
		assertNotNull(rs);
		assertEquals(1, rs.getResources().size());

		repository.detach(person);
		assertNull(person.eResource());
		assertEquals(0, rs.getResources().size());

		Person personResult = repository.getEObject(TestPackage.Literals.PERSON, "test");
		assertNotNull(personResult);
		assertNotEquals(person, personResult);
		assertNotEquals(r, personResult.eResource());

		assertTrue(EcoreUtil.equals(person, personResult));

		repositoryConfig.delete();
		repositoryConfig = null;
		
		Thread.sleep(2000l);

		repository = getService(2000l, "(" + EMFRepository.PROP_ID + "=" + clientId + ")");
		assertNull(repository);
		dbProvider = getService(MongoDatabaseProvider.class, 2000);
		assertNull(dbProvider);
		clientProvider = getService(MongoClientProvider.class, 2000);
		assertNull(clientProvider);
	}

	@Test
	public void testEMFMongoRepositoryPrototype() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {
		
		// create mongo client configuration
		ServiceReference<ConfigurationAdmin> caRef = context.getServiceReference(ConfigurationAdmin.class);
		assertNotNull(caRef);
		ConfigurationAdmin cm = context.getService(caRef);
		assertNotNull(cm);
		
		repositoryConfig = cm.createFactoryConfiguration(EMFMongoConfiguratorConstants.EMF_MONGO_REPOSITORY_CONFIGURATOR_CONFIGURATION_NAME, "?");
		assertNotNull(repositoryConfig);
		
		// has to be a new configuration
		assertNull(repositoryConfig.getProperties());
		
		/**
		 * mongo.instances=test1
		 * test1.baseUris=mongodb://localhost
		 * test1.databases=test
		 */
		
		Dictionary<String, Object> configProperties = new Hashtable<>();
		configProperties.put("mongo.instances", "test1");
		configProperties.put("test1.baseUris", "mongodb://" + mongoHost);
		configProperties.put("test1.databases", "test");
		configProperties.put("test1." + EMFMongoConfiguratorConstants.MONGO_REPOSITORY_TYPE, EMFMongoConfiguratorConstants.Type.PROTOTYPE.toString());
		
		
		repositoryConfig.update(configProperties);
		
		String clientId = "test1.test";
		
		MongoClientProvider clientProvider = getService(MongoClientProvider.class, 5000);
		assertNotNull(clientProvider);
		
		MongoDatabaseProvider dbProvider = getService(MongoDatabaseProvider.class, 5000);
		assertNotNull(dbProvider);
		
		ResourceSetFactory rsf = getService(ResourceSetFactory.class, 5000l);
		assertNotNull(rsf);
		
		ServiceReference<EMFRepository> repositoryRef = getServiceReference(5000l, "(" + EMFRepository.PROP_ID + "=" + clientId + ")");
		assertNotNull(repositoryRef);
		
		ServiceObjects<EMFRepository> serviceObjects = context.getServiceObjects(repositoryRef);
		
		EMFRepository repository1 = serviceObjects.getService();
		EMFRepository repository2 = serviceObjects.getService();
		assertNotEquals(repository1, repository2);
	}

	@Test
	public void testVareplacement() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {
		
		// create mongo client configuration
		ServiceReference<ConfigurationAdmin> caRef = context.getServiceReference(ConfigurationAdmin.class);
		assertNotNull(caRef);
		ConfigurationAdmin cm = context.getService(caRef);
		assertNotNull(cm);
		
		repositoryConfig = cm.createFactoryConfiguration(EMFMongoConfiguratorConstants.EMF_MONGO_REPOSITORY_CONFIGURATOR_CONFIGURATION_NAME, "?");
		assertNotNull(repositoryConfig);
		
		// has to be a new configuration
		assertNull(repositoryConfig.getProperties());
		
		/**
		 * mongo.instances=test1
		 * test1.baseUris=mongodb://localhost
		 * test1.databases=test
		 */
		
		Dictionary<String, Object> configProperties = new Hashtable<>();
		configProperties.put("mongo.instances", "test1");
		configProperties.put("test1.baseUris", "mongodb://" + mongoHost);
		configProperties.put("test1.baseUris.env", "URI_ENV");
		configProperties.put("test1.databases", "test");
		configProperties.put("test1.test.user", "test");
		configProperties.put("test1.test.user.env", "USER_ENV");
		configProperties.put("test1.test.password", "1234");
		configProperties.put("test1.test.password.env", "PWD_ENV");
		
		System.setProperty("USER_ENV", "envUser");
		System.setProperty("URI_ENV", "mongodb://127.0.0.1");
		System.setProperty("PWD_ENV", "testPwd");
		
		repositoryConfig.update(configProperties);
		
		
		ServiceReference<MongoClientProvider> clientProvider = getServiceReference(MongoClientProvider.class, 5000);
		assertNotNull(clientProvider);
		
		assertEquals(System.getProperty("URI_ENV"), clientProvider.getProperty(MongoClientProvider.PROP_URI));
		String property = (String) clientProvider.getProperty(MongoClientProvider.PROP_CREDENTIALS);
		assertNotNull(property);
		assertTrue(property.contains(System.getProperty("USER_ENV")));
		assertTrue(property.contains(System.getProperty("PWD_ENV")));
		
		ServiceReference<MongoDatabaseProvider> dbProvider = getServiceReference(MongoDatabaseProvider.class, 5000);
		assertNotNull(dbProvider);
		
	}

	<T> T getService(long timeout, String filter) throws InterruptedException, InvalidSyntaxException {
		Filter f = FrameworkUtil.createFilter(filter);
		ServiceTracker<T, T> tracker = new ServiceTracker<>(context, f, null);
		tracker.open();
		try {
			return tracker.waitForService(timeout);
		} catch (Exception e) {
			return null;
		} finally {
			tracker.close();
		}
	}

	 <T> ServiceReference<T>getServiceReference(long timeout, String filter) throws InterruptedException, InvalidSyntaxException {
		Filter f = FrameworkUtil.createFilter(filter);
		ServiceTracker<T, T> tracker = new ServiceTracker<>(context, f, null);
		tracker.open();
		try {
			if(tracker.waitForService(timeout) != null) {
				return tracker.getServiceReference(); 
			} else {
				return null;
			}
		} catch (Exception e) {
			return null;
		} finally {
			tracker.close();
		}
	}

	<T> T getService(Class<T> clazz, long timeout) throws InterruptedException, InvalidSyntaxException {
		ServiceTracker<T, T> tracker = new ServiceTracker<>(context, clazz, null);
		tracker.open();
		try {
			return tracker.waitForService(timeout);
		} catch (Exception e) {
			return null;
		} finally {
			tracker.close();
		}
	}
	
	 <T> ServiceReference<T> getServiceReference(Class<T> clazz, long timeout) throws InterruptedException, InvalidSyntaxException {
		ServiceTracker<T, T> tracker = new ServiceTracker<>(context, clazz, null);
		tracker.open();
		try {
			if(tracker.waitForService(timeout) != null) {
				return tracker.getServiceReference(); 
			} else {
				return null;
			}
		} catch (Exception e) {
			return null;
		} finally {
			tracker.close();
		}
	}

}
