/**
 * Copyright (c) 2012 - 2017 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.emf.repository.mongo.tests;


import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.emf.osgi.EPackageConfigurator;
import org.gecko.emf.osgi.ResourceFactoryConfigurator;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.emf.osgi.model.test.Person;
import org.gecko.emf.osgi.model.test.TestFactory;
import org.gecko.emf.osgi.model.test.TestPackage;
import org.gecko.emf.osgi.model.test.configurator.TestPackageConfigurator;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.mongo.annotations.RequireMongoEMFRepository;
import org.gecko.emf.repository.mongo.api.EMFMongoConfiguratorConstants;
import org.gecko.mongo.osgi.MongoClientProvider;
import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.gecko.util.test.AbstractOSGiTest;
import org.gecko.util.test.ServiceChecker;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.BundleException;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceObjects;
import org.osgi.framework.ServiceReference;
import org.osgi.service.cm.Configuration;

import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;

/**
 * Integration tests for the complete EMF mongo repository setup
 * @author Mark Hoffmann
 * @since 26.07.2017
 */
@RequireMongoEMFRepository
@RunWith(MockitoJUnitRunner.class)
public class MongoConfiguratorIntegrationTest extends AbstractOSGiTest{

	private MongoClient client;
	private MongoCollection<?> collection;
	private String mongoHost = System.getProperty("mongo.host", "localhost");

	public MongoConfiguratorIntegrationTest() {
		super(FrameworkUtil.getBundle(MongoRepositoryIntegrationTest.class).getBundleContext());
	}

	public void doBefore(){
		registerServiceForCleanup(new TestPackageConfigurator(), new Hashtable<String, Object>(), EPackageConfigurator.class.getName(), ResourceFactoryConfigurator.class.getName());
		MongoClientOptions options = MongoClientOptions.builder().build();
		client = new MongoClient(mongoHost, options);
	}

	public void doAfter() {
		if (collection != null) {
			collection.drop();
		}
		if (client != null) {
			client.close();
		}
	}

	@Test
	public void testEMFMongoRepository() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {
		/**
		 * mongo.instances=test1
		 * test1.baseUris=mongodb://localhost
		 * test1.databases=test
		 */

		Dictionary<String, Object> configProperties = new Hashtable<>();
		configProperties.put("mongo.instances", "test1");
		configProperties.put("test1.baseUris", "mongodb://" + mongoHost);
		configProperties.put("test1.databases", "test");

		String clientId = "test1.test";

		ServiceChecker<EMFRepository> repoTracker = createdCheckerTrackedForCleanUp("(" + EMFRepository.PROP_ID + "=" + clientId + ")");
		ServiceChecker<MongoClientProvider> clientTracker = createCheckerTrackedForCleanUp(MongoClientProvider.class);
		ServiceChecker<MongoDatabaseProvider> dbTracker = createCheckerTrackedForCleanUp(MongoDatabaseProvider.class);
		ServiceChecker<ResourceSetFactory> rsfTracker = createCheckerTrackedForCleanUp(ResourceSetFactory.class);
		
		repoTracker.start();
		clientTracker.start();
		dbTracker.start();
		rsfTracker.start();
		
		Configuration repositoryConfig = createConfigForCleanup(EMFMongoConfiguratorConstants.EMF_MONGO_REPOSITORY_CONFIGURATOR_CONFIGURATION_NAME, "?", configProperties);

		assertTrue(repoTracker.awaitCreation());
		assertTrue(clientTracker.awaitCreation());
		assertTrue(dbTracker.awaitCreation());
		assertTrue(rsfTracker.awaitCreation());

		EMFRepository repository = getService(FrameworkUtil.createFilter("(" + EMFRepository.PROP_ID + "=" + clientId + ")"), 5000);

		Person person = TestFactory.eINSTANCE.createPerson();
		person.setId("test");
		person.setFirstName("Emil");
		person.setLastName("Tester");
		URI uri = repository.createUri(person);
		assertEquals("mongodb://test1/test/Person/test", uri.toString());

		MongoDatabase database = client.getDatabase("test");
		collection = database.getCollection("Person"); 
		collection.drop();

		assertEquals(0, collection.countDocuments());

		CountDownLatch latch = new CountDownLatch(1);
		latch.await(1, TimeUnit.SECONDS);
		
		repository.save(person);

		assertEquals(1, collection.countDocuments());

		Resource r = person.eResource();
		assertNotNull(r);
		ResourceSet rs = r.getResourceSet();
		assertNotNull(rs);
		assertEquals(1, rs.getResources().size());

		repository.detach(person);
		assertNull(person.eResource());
		assertEquals(0, rs.getResources().size());

		Person personResult = repository.getEObject(TestPackage.Literals.PERSON, "test");
		assertNotNull(personResult);
		assertNotEquals(person, personResult);
		assertNotEquals(r, personResult.eResource());

		assertTrue(EcoreUtil.equals(person, personResult));

		deleteConfigurationAndRemoveFromCleanup(repositoryConfig);

		assertTrue(repoTracker.awaitRemoval());
		assertTrue(dbTracker.awaitRemoval());
		assertTrue(clientTracker.awaitRemoval());
	}

	@Test
	public void testEMFMongoRepositoryPrototypeInstance() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {
		
		/**
		 * mongo.instances=test1
		 * test1.baseUris=mongodb://localhost
		 * test1.databases=test
		 */

		Dictionary<String, Object> configProperties = new Hashtable<>();
		configProperties.put("mongo.instances", "test1");
		configProperties.put("test1.baseUris", "mongodb://" + mongoHost);
		configProperties.put("test1.databases", "test, test2");
		configProperties.put("test1." + EMFMongoConfiguratorConstants.MONGO_REPOSITORY_TYPE, EMFMongoConfiguratorConstants.Type.PROTOTYPE.toString());

		String clientId1 = "test1.test";
		String clientId2 = "test1.test2";

		ServiceChecker<EMFRepository> repoTracker = createdCheckerTrackedForCleanUp("(" + EMFRepository.PROP_ID + "=" + clientId1 + ")");
		ServiceChecker<EMFRepository> repo2Tracker = createdCheckerTrackedForCleanUp("(" + EMFRepository.PROP_ID + "=" + clientId2 + ")");
		ServiceChecker<MongoClientProvider> clientTracker = createCheckerTrackedForCleanUp(MongoClientProvider.class);
		ServiceChecker<MongoDatabaseProvider> dbTracker = createCheckerTrackedForCleanUp(MongoDatabaseProvider.class);
		ServiceChecker<ResourceSetFactory> rsfTracker = createCheckerTrackedForCleanUp(ResourceSetFactory.class);

		repoTracker.start();
		repo2Tracker.start();
		clientTracker.start();
		dbTracker.setCreateExpectationCount(2);
		dbTracker.start();
		rsfTracker.start();
		
		Configuration configuration = createConfigForCleanup(EMFMongoConfiguratorConstants.EMF_MONGO_REPOSITORY_CONFIGURATOR_CONFIGURATION_NAME, "?", configProperties);

		assertTrue(repoTracker.awaitCreation());
		assertTrue(repo2Tracker.awaitCreation());
		assertTrue(clientTracker.awaitCreation());
		assertTrue(dbTracker.awaitCreation());
		assertTrue(rsfTracker.awaitCreation());

		ServiceObjects<EMFRepository> repo1ServiceObjects = getServiceObjects(EMFRepository.class,"(" + EMFRepository.PROP_ID + "=" + clientId1 + ")");

		
		EMFRepository repository1 = repo1ServiceObjects.getService();
		EMFRepository repository2 = repo1ServiceObjects.getService();
		assertNotEquals(repository1, repository2);

		repo1ServiceObjects.ungetService(repository1);
		repo1ServiceObjects.ungetService(repository2);
		
		ServiceObjects<EMFRepository> repo2ServiceObjects = getServiceObjects(EMFRepository.class,"(" + EMFRepository.PROP_ID + "=" + clientId2 + ")");
		
		
		repository1 = repo2ServiceObjects.getService();
		repository2 = repo2ServiceObjects.getService();
		assertNotEquals(repository1, repository2);
		
		repo2ServiceObjects.ungetService(repository1);
		repo2ServiceObjects.ungetService(repository2);
		
		deleteConfigurationAndRemoveFromCleanup(configuration);
	}

	@Test
	public void testEMFMongoRepositoryPrototypedbLevel() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {
	
		
		/**
		 * mongo.instances=test1
		 * test1.baseUris=mongodb://localhost
		 * test1.databases=test
		 */

		Dictionary<String, Object> configProperties = new Hashtable<>();
		configProperties.put("mongo.instances", "test1");
		configProperties.put("test1.baseUris", "mongodb://" + mongoHost);
		configProperties.put("test1.databases", "test, test2");
		configProperties.put("test1.test." + EMFMongoConfiguratorConstants.MONGO_REPOSITORY_TYPE, EMFMongoConfiguratorConstants.Type.PROTOTYPE.toString());
		configProperties.put("test1.test2." + EMFMongoConfiguratorConstants.MONGO_REPOSITORY_TYPE, EMFMongoConfiguratorConstants.Type.SINGLETON.toString());

		String clientId1 = "test1.test";
		String clientId2 = "test1.test2";

		ServiceChecker<EMFRepository> repoTracker = createdCheckerTrackedForCleanUp("(" + EMFRepository.PROP_ID + "=" + clientId1 + ")");
		ServiceChecker<EMFRepository> repo2Tracker = createdCheckerTrackedForCleanUp("(" + EMFRepository.PROP_ID + "=" + clientId2 + ")");
		ServiceChecker<MongoClientProvider> clientTracker = createCheckerTrackedForCleanUp(MongoClientProvider.class);
		ServiceChecker<MongoDatabaseProvider> dbTracker = createCheckerTrackedForCleanUp(MongoDatabaseProvider.class);
		ServiceChecker<ResourceSetFactory> rsfTracker = createCheckerTrackedForCleanUp(ResourceSetFactory.class);

		repoTracker.start();
		repo2Tracker.start();
		clientTracker.start();
		dbTracker.setCreateExpectationCount(2);
		dbTracker.start();
		rsfTracker.start();
		
		Configuration configuration = createConfigForCleanup(EMFMongoConfiguratorConstants.EMF_MONGO_REPOSITORY_CONFIGURATOR_CONFIGURATION_NAME, "?", configProperties);

		assertTrue(repoTracker.awaitCreation());
		assertTrue(repo2Tracker.awaitCreation());
		assertTrue(clientTracker.awaitCreation());
		assertTrue(dbTracker.awaitCreation());
		assertTrue(rsfTracker.awaitCreation());

		ServiceObjects<EMFRepository> repo1ServiceObjects = getServiceObjects(EMFRepository.class,"(" + EMFRepository.PROP_ID + "=" + clientId1 + ")");

		
		EMFRepository repository1 = repo1ServiceObjects.getService();
		EMFRepository repository2 = repo1ServiceObjects.getService();
		assertNotEquals(repository1, repository2);

		repo1ServiceObjects.ungetService(repository1);
		repo1ServiceObjects.ungetService(repository2);
		
		ServiceObjects<EMFRepository> repo2ServiceObjects = getServiceObjects(EMFRepository.class,"(" + EMFRepository.PROP_ID + "=" + clientId2 + ")");
		
		
		repository1 = repo2ServiceObjects.getService();
		repository2 = repo2ServiceObjects.getService();
		assertEquals(repository1, repository2);
		
		repo2ServiceObjects.ungetService(repository1);
		repo2ServiceObjects.ungetService(repository2);
		
		deleteConfigurationAndRemoveFromCleanup(configuration);
	}

	@Test
	public void testVarReplacementDBAuth() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {
		registerServiceForCleanup(new TestPackageConfigurator(), new Hashtable<String, Object>(), EPackageConfigurator.class.getName(), ResourceFactoryConfigurator.class.getName());
	
		/**
		 * mongo.instances=test1
		 * test1.baseUris=mongodb://localhost
		 * test1.databases=test
		 */
		
		Dictionary<String, Object> configProperties = new Hashtable<>();
		configProperties.put("mongo.instances", "test1");
		configProperties.put("test1.baseUris", "mongodb://" + mongoHost);
		configProperties.put("test1.baseUris.env", "URI_ENV");
		configProperties.put("test1.databases", "test");
		configProperties.put("test1.test.user", "test");
		configProperties.put("test1.test.user.env", "USER_ENV");
		configProperties.put("test1.test.password", "1234");
		configProperties.put("test1.test.password.env", "PWD_ENV");
		
		System.setProperty("USER_ENV", "envUser");
		System.setProperty("URI_ENV", "mongodb://127.0.0.1");
		System.setProperty("PWD_ENV", "testPwd");
		
		String clientId = "test1.test";

		ServiceChecker<EMFRepository> repoTracker = createdCheckerTrackedForCleanUp("(" + EMFRepository.PROP_ID + "=" + clientId + ")");
		ServiceChecker<MongoClientProvider> clientTracker = createCheckerTrackedForCleanUp(MongoClientProvider.class);
		ServiceChecker<MongoDatabaseProvider> dbTracker = createCheckerTrackedForCleanUp(MongoDatabaseProvider.class);
		ServiceChecker<ResourceSetFactory> rsfTracker = createCheckerTrackedForCleanUp(ResourceSetFactory.class);
		
		repoTracker.start();
		clientTracker.start();
		dbTracker.start();
		rsfTracker.start();
		
		Configuration configuration = createConfigForCleanup(EMFMongoConfiguratorConstants.EMF_MONGO_REPOSITORY_CONFIGURATOR_CONFIGURATION_NAME, "?", configProperties);

		assertTrue(repoTracker.awaitCreation());
		assertTrue(clientTracker.awaitCreation());
		assertTrue(dbTracker.awaitCreation());
		assertTrue(rsfTracker.awaitCreation());
		
		ServiceReference<MongoClientProvider> clientProvider = getServiceReference(MongoClientProvider.class);
		
		assertEquals(System.getProperty("URI_ENV"), clientProvider.getProperty(MongoClientProvider.PROP_URI));
		String property = (String) clientProvider.getProperty(MongoClientProvider.PROP_CREDENTIALS);
		assertNotNull(property);
		assertTrue(property.contains(System.getProperty("USER_ENV")));
		assertTrue(property.contains(System.getProperty("PWD_ENV")));
		
		ServiceReference<MongoDatabaseProvider> dbProvider = getServiceReference(MongoDatabaseProvider.class);
		assertNotNull(dbProvider);
		deleteConfigurationAndRemoveFromCleanup(configuration);
	}
	
	@Test
	public void testVarReplacementInstanceAuth() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {
		registerServiceForCleanup(new TestPackageConfigurator(), new Hashtable<String, Object>(), EPackageConfigurator.class.getName(), ResourceFactoryConfigurator.class.getName());
		
		/**
		 * mongo.instances=test1
		 * test1.baseUris=mongodb://localhost
		 * test1.databases=test
		 */
		
		Dictionary<String, Object> configProperties = new Hashtable<>();
		String URI_ENV_NAME = "URI_ENV";
		String AUTH_SOURCE_ENV_NAME = "AUTH_SRC_ENV";
		String USER_ENV_NAME = "USER_ENV";
		String PWD_ENV_NAME = "PWD_ENV";
		configProperties.put("mongo.instances", "test1");
		configProperties.put("test1.baseUris", "mongodb://" + mongoHost);
		configProperties.put("test1.baseUris.env", URI_ENV_NAME);
		configProperties.put("test1.databases", "test");
		configProperties.put("test1.authSource", "test");
		configProperties.put("test1.authSource.env", AUTH_SOURCE_ENV_NAME);
		configProperties.put("test1.user", "test");
		configProperties.put("test1.user.env", USER_ENV_NAME);
		configProperties.put("test1.password", "1234");
		configProperties.put("test1.password.env", PWD_ENV_NAME);
		
		System.setProperty(USER_ENV_NAME, "envUser");
		System.setProperty(URI_ENV_NAME, "mongodb://127.0.0.1");
		System.setProperty(PWD_ENV_NAME, "envPwd");
		System.setProperty(AUTH_SOURCE_ENV_NAME, "envSource");
		
		String clientId = "test1.test";
		
		ServiceChecker<EMFRepository> repoTracker = createdCheckerTrackedForCleanUp("(" + EMFRepository.PROP_ID + "=" + clientId + ")");
		ServiceChecker<MongoClientProvider> clientTracker = createCheckerTrackedForCleanUp(MongoClientProvider.class);
		ServiceChecker<MongoDatabaseProvider> dbTracker = createCheckerTrackedForCleanUp(MongoDatabaseProvider.class);
		ServiceChecker<ResourceSetFactory> rsfTracker = createCheckerTrackedForCleanUp(ResourceSetFactory.class);
		
		repoTracker.start();
		clientTracker.start();
		dbTracker.start();
		rsfTracker.start();
		
		Configuration configuration = createConfigForCleanup(EMFMongoConfiguratorConstants.EMF_MONGO_REPOSITORY_CONFIGURATOR_CONFIGURATION_NAME, "?", configProperties);
		
		assertTrue(repoTracker.awaitCreation());
		assertTrue(clientTracker.awaitCreation());
		assertTrue(dbTracker.awaitCreation());
		assertTrue(rsfTracker.awaitCreation());
		
		ServiceReference<MongoClientProvider> clientProvider = getServiceReference(MongoClientProvider.class);
		
		assertEquals(System.getProperty(URI_ENV_NAME), clientProvider.getProperty(MongoClientProvider.PROP_URI));
		String property = (String) clientProvider.getProperty(MongoClientProvider.PROP_CREDENTIALS);
		assertNotNull(property);
		assertTrue(property.contains(System.getProperty(USER_ENV_NAME)));
		assertTrue(property.contains(System.getProperty(PWD_ENV_NAME)));
		assertTrue(property.contains(System.getProperty(AUTH_SOURCE_ENV_NAME)));
		
		ServiceReference<MongoDatabaseProvider> dbProvider = getServiceReference(MongoDatabaseProvider.class);
		assertNotNull(dbProvider);
		
		deleteConfigurationAndRemoveFromCleanup(configuration);
	}
	
	@Test
	public void testVarReplacementInstanceAuthOverDBAuth() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {
		registerServiceForCleanup(new TestPackageConfigurator(), new Hashtable<String, Object>(), EPackageConfigurator.class.getName(), ResourceFactoryConfigurator.class.getName());
		
		/**
		 * mongo.instances=test1
		 * test1.baseUris=mongodb://localhost
		 * test1.databases=test
		 */
		
		Dictionary<String, Object> configProperties = new Hashtable<>();
		String URI_ENV_NAME = "URI_ENV";
		String AUTH_SOURCE_ENV_NAME = "AUTH_SRC_ENV";
		String USER_ENV_NAME = "USER_ENV";
		String PWD_ENV_NAME = "PWD_ENV";
		String DB_USER_ENV_NAME = "DB_USER_ENV";
		String DB_PWD_ENV_NAME = "DB_PWD_ENV";
		configProperties.put("mongo.instances", "test1");
		configProperties.put("test1.baseUris", "mongodb://" + mongoHost);
		configProperties.put("test1.baseUris.env", URI_ENV_NAME);
		configProperties.put("test1.databases", "test");
		configProperties.put("test1.authSource", "test");
		configProperties.put("test1.authSource.env", AUTH_SOURCE_ENV_NAME);
		configProperties.put("test1.user", "test");
		configProperties.put("test1.user.env", USER_ENV_NAME);
		configProperties.put("test1.password", "1234");
		configProperties.put("test1.password.env", PWD_ENV_NAME);
		configProperties.put("test1.test.user", "test");
		configProperties.put("test1.test.user.env", DB_USER_ENV_NAME);
		configProperties.put("test1.test.password", "1234");
		configProperties.put("test1.test.password.env", DB_PWD_ENV_NAME);
		
		System.setProperty(USER_ENV_NAME, "envUser");
		System.setProperty(URI_ENV_NAME, "mongodb://127.0.0.1");
		System.setProperty(PWD_ENV_NAME, "envPwd");
		System.setProperty(AUTH_SOURCE_ENV_NAME, "envSource");
		
		String clientId = "test1.test";
		
		ServiceChecker<EMFRepository> repoTracker = createdCheckerTrackedForCleanUp("(" + EMFRepository.PROP_ID + "=" + clientId + ")");
		ServiceChecker<MongoClientProvider> clientTracker = createCheckerTrackedForCleanUp(MongoClientProvider.class);
		ServiceChecker<MongoDatabaseProvider> dbTracker = createCheckerTrackedForCleanUp(MongoDatabaseProvider.class);
		ServiceChecker<ResourceSetFactory> rsfTracker = createCheckerTrackedForCleanUp(ResourceSetFactory.class);
		
		repoTracker.start();
		clientTracker.start();
		dbTracker.start();
		rsfTracker.start();
		
		Configuration configuration = createConfigForCleanup(EMFMongoConfiguratorConstants.EMF_MONGO_REPOSITORY_CONFIGURATOR_CONFIGURATION_NAME, "?", configProperties);
		
		assertTrue(repoTracker.awaitCreation());
		assertTrue(clientTracker.awaitCreation());
		assertTrue(dbTracker.awaitCreation());
		assertTrue(rsfTracker.awaitCreation());
		
		ServiceReference<MongoClientProvider> clientProvider = getServiceReference(MongoClientProvider.class);
		
		assertEquals(System.getProperty(URI_ENV_NAME), clientProvider.getProperty(MongoClientProvider.PROP_URI));
		String property = (String) clientProvider.getProperty(MongoClientProvider.PROP_CREDENTIALS);
		assertNotNull(property);
		assertTrue(property.contains(System.getProperty(USER_ENV_NAME)));
		assertTrue(property.contains(System.getProperty(PWD_ENV_NAME)));
		assertTrue(property.contains(System.getProperty(AUTH_SOURCE_ENV_NAME)));
		
		ServiceReference<MongoDatabaseProvider> dbProvider = getServiceReference(MongoDatabaseProvider.class);
		assertNotNull(dbProvider);
		
		deleteConfigurationAndRemoveFromCleanup(configuration);
	}
}
