/**
 * Copyright (c) 2012 - 2017 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.emf.repository.mongo.tests;


import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.util.Collections;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.emf.mongo.Options;
import org.gecko.emf.osgi.EPackageConfigurator;
import org.gecko.emf.osgi.ResourceFactoryConfigurator;
import org.gecko.emf.osgi.model.test.Address;
import org.gecko.emf.osgi.model.test.Person;
import org.gecko.emf.osgi.model.test.TestFactory;
import org.gecko.emf.osgi.model.test.TestPackage;
import org.gecko.emf.osgi.model.test.configurator.TestPackageConfigurator;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.mongo.api.EMFMongoConfiguratorConstants;
import org.gecko.emf.repository.query.IQueryBuilder;
import org.gecko.emf.repository.query.QueryRepository;
import org.gecko.mongo.osgi.MongoClientProvider;
import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.gecko.mongo.osgi.configuration.ConfigurationProperties;
import org.gecko.util.test.AbstractOSGiTest;
import org.gecko.util.test.ServiceChecker;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.BundleException;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.service.cm.Configuration;

import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;

/**
 * Integration tests for the complete EMF mongo repository setup
 * @author Mark Hoffmann
 * @since 26.07.2017
 */
@RunWith(MockitoJUnitRunner.class)
public class MongoRepositoryIntegrationTest extends AbstractOSGiTest{


	private MongoClient client;
	private MongoCollection<?> collection;
	private String mongoHost = System.getProperty("mongo.host", "localhost");

	public MongoRepositoryIntegrationTest() {
		super(FrameworkUtil.getBundle(MongoRepositoryIntegrationTest.class).getBundleContext());
	}

	public void doBefore(){
		MongoClientOptions options = MongoClientOptions.builder().build();
		client = new MongoClient(mongoHost, options);
		registerServiceForCleanup(new TestPackageConfigurator(), new Hashtable<String, Object>(), EPackageConfigurator.class.getName(), ResourceFactoryConfigurator.class.getName());
	}

	public void doAfter() {
		if (collection != null) {
			collection.drop();
		}
		if (client != null) {
			client.close();
		}
	}

	@Test
	public void testEMFMongoRepository() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {

		// add service properties
		String clientId = "testClient";
		String clientUri = "mongodb://" + mongoHost + ":27017";
		Dictionary<String, Object> p = new Hashtable<String, Object>();
		p.put(MongoClientProvider.PROP_CLIENT_ID, clientId);
		p.put(MongoClientProvider.PROP_URI, clientUri);

		ServiceChecker<MongoClientProvider> clientChecker = createCheckerTrackedForCleanUp(MongoClientProvider.class);
		clientChecker.start();
		createConfigForCleanup(ConfigurationProperties.CLIENT_PID, "?", p);
		
		assertTrue(clientChecker.awaitCreation());
		
		// add service properties
		String dbAlias = "testDB";
		String db = "test";
		Dictionary<String, Object> dbp = new Hashtable<String, Object>();
		dbp.put(MongoDatabaseProvider.PROP_ALIAS, dbAlias);
		dbp.put(MongoDatabaseProvider.PROP_DATABASE, db);
		
		ServiceChecker<MongoDatabaseProvider> dbChecker = createCheckerTrackedForCleanUp(MongoDatabaseProvider.class);
		dbChecker.start();

		Configuration dbConfig = createConfigForCleanup(ConfigurationProperties.DATABASE_PID, "?", dbp);

		assertTrue(dbChecker.awaitCreation());
		
		Dictionary<String, Object> properties = new Hashtable<>();

		String repoId = "test_repo";
		properties.put(EMFRepository.PROP_ID, repoId);
		properties.put(EMFRepository.PROP_BASE_URI, clientUri + "/" + db);
		
		ServiceChecker<EMFRepository> repoChecker = createCheckerTrackedForCleanUp(EMFRepository.class);
		repoChecker.start();
		
		createConfigForCleanup(EMFMongoConfiguratorConstants.SINGLETON_REPOSITORY_CONFIGURATION_NAME, "?", properties);
		
		assertTrue(repoChecker.awaitCreation());
		
		MongoDatabase database = client.getDatabase(db);
		collection = database.getCollection("Person"); 
		collection.drop();
		
		EMFRepository repository = getService(FrameworkUtil.createFilter("(" + EMFRepository.PROP_ID + "=" + repoId + ")"), 5000l);

		Person person = TestFactory.eINSTANCE.createPerson();
		person.setId("test");
		person.setFirstName("Emil");
		person.setLastName("Tester");
		URI uri = repository.createUri(person);
		assertEquals(clientUri + "/" + db + "/Person/test", uri.toString());
		assertEquals(0, collection.countDocuments());
		
		repository.save(person);
		
		assertEquals(1, collection.countDocuments());
		
		Resource r = person.eResource();
		assertNotNull(r);
		ResourceSet rs = r.getResourceSet();
		assertNotNull(rs);
		assertEquals(1, rs.getResources().size());

		repository.detach(person);
		assertNull(person.eResource());
		assertEquals(0, rs.getResources().size());

		Person personResult = repository.getEObject(TestPackage.Literals.PERSON, "test");
		assertNotNull(personResult);
		assertNotEquals(person, personResult);
		assertNotEquals(r, personResult.eResource());

		assertTrue(EcoreUtil.equals(person, personResult));
		
		deleteConfigurationAndRemoveFromCleanup(dbConfig);
	}

	@Test
	public void testMongoEMFMongoRepositoryGetByQuery() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {
		
		// add service properties
		String clientId = "testClient";
		String clientUri = "mongodb://" + mongoHost + ":27017";
		Dictionary<String, Object> p = new Hashtable<String, Object>();
		p.put(MongoClientProvider.PROP_CLIENT_ID, clientId);
		p.put(MongoClientProvider.PROP_URI, clientUri);
		
		ServiceChecker<MongoClientProvider> clientChecker = createCheckerTrackedForCleanUp(MongoClientProvider.class);
		clientChecker.start();
		createConfigForCleanup(ConfigurationProperties.CLIENT_PID, "?", p);
		
		assertTrue(clientChecker.awaitCreation());
		
		// add service properties
		String dbAlias = "testDB";
		String db = "test";
		Dictionary<String, Object> dbp = new Hashtable<String, Object>();
		dbp.put(MongoDatabaseProvider.PROP_ALIAS, dbAlias);
		dbp.put(MongoDatabaseProvider.PROP_DATABASE, db);
		
		ServiceChecker<MongoDatabaseProvider> dbChecker = createCheckerTrackedForCleanUp(MongoDatabaseProvider.class);
		dbChecker.start();
		
		Configuration dbConfig = createConfigForCleanup(ConfigurationProperties.DATABASE_PID, "?", dbp);
		
		assertTrue(dbChecker.awaitCreation());
		
		Dictionary<String, Object> properties = new Hashtable<>();
		
		String repoId = "test_repo";
		properties.put(EMFRepository.PROP_ID, repoId);
		properties.put(EMFRepository.PROP_BASE_URI, clientUri + "/" + db);
		
		ServiceChecker<EMFRepository> repoChecker = createCheckerTrackedForCleanUp(EMFRepository.class);
		repoChecker.start();
		
		createConfigForCleanup(EMFMongoConfiguratorConstants.SINGLETON_REPOSITORY_CONFIGURATION_NAME, "?", properties);
		
		assertTrue(repoChecker.awaitCreation());
		
		MongoDatabase database = client.getDatabase(db);
		collection = database.getCollection("Person"); 
		collection.drop();
		
		EMFRepository repository = getService(FrameworkUtil.createFilter("(" + EMFRepository.PROP_ID + "=" + repoId + ")"), 5000l);
		QueryRepository queryRepo = (QueryRepository) repository.getAdapter(QueryRepository.class);
		Person person = TestFactory.eINSTANCE.createPerson();
		person.setId("test");
		person.setFirstName("Emil");
		person.setLastName("Tester");
		URI uri = repository.createUri(person);
		assertEquals(clientUri + "/" + db + "/Person/test", uri.toString());
		assertEquals(0, collection.countDocuments());
		
		repository.save(person);

		assertEquals(1, collection.countDocuments());

		person = TestFactory.eINSTANCE.createPerson();
		person.setId("test2");
		person.setFirstName("Emil2");
		person.setLastName("Tester2");
		uri = repository.createUri(person);
		assertEquals(clientUri + "/" + db + "/Person/test2", uri.toString());
		
		repository.save(person);
		
		assertEquals(2, collection.countDocuments());
		
		ResourceSet resourceSet = repository.getResourceSet();
		
		resourceSet.getResources().clear();
		
		assertTrue(resourceSet.getResources().isEmpty());
		
		IQueryBuilder allQuery = queryRepo.createQueryBuilder().allQuery();
		
		List<EObject> result = queryRepo.getEObjectsByQuery(TestPackage.Literals.PERSON, allQuery.build());
		
		//only the result of our 2 saved Persons must have a related resource
		assertEquals(2, result.size());
		assertEquals(result.size(), resourceSet.getResources().size());
		
		resourceSet.getResources().clear();
		
		assertTrue(resourceSet.getResources().isEmpty());
		
		result = queryRepo.getEObjectsByQuery(TestPackage.Literals.PERSON, allQuery.build(), Collections.singletonMap(Options.OPTION_LAZY_RESULT_LOADING, true));
		
		// We load lazy, so no resource must remain before after the query
		assertEquals(2, result.size());
		assertEquals(0, resourceSet.getResources().size());
		
		BasicEList<EObject> basicList = (BasicEList<EObject>) result;
		
		EObject proxyPerson = basicList.basicGet(0);
		
		assertTrue(proxyPerson.eIsProxy());
		assertEquals(0, resourceSet.getResources().size());
		
		person = (Person) basicList.get(0);
		
		assertFalse(person.eIsProxy());
		assertEquals(1, resourceSet.getResources().size());
		assertEquals(person, basicList.basicGet(0));
		
		
		deleteConfigurationAndRemoveFromCleanup(dbConfig);
	}

	@Test
	public void testEMFMongoRepositorySaveMultiple() throws BundleException, InvalidSyntaxException, IOException, InterruptedException {
		
		// add service properties
		String clientId = "testClient";
		String clientUri = "mongodb://" + mongoHost + ":27017";
		Dictionary<String, Object> p = new Hashtable<String, Object>();
		p.put(MongoClientProvider.PROP_CLIENT_ID, clientId);
		p.put(MongoClientProvider.PROP_URI, clientUri);

		ServiceChecker<MongoClientProvider> clientChecker = createCheckerTrackedForCleanUp(MongoClientProvider.class);
		clientChecker.start();
		createConfigForCleanup(ConfigurationProperties.CLIENT_PID, "?", p);
		
		assertTrue(clientChecker.awaitCreation());

		// add service properties
		String dbAlias = "testDB";
		String db = "test";
		Dictionary<String, Object> dbp = new Hashtable<String, Object>();
		dbp.put(MongoDatabaseProvider.PROP_ALIAS, dbAlias);
		dbp.put(MongoDatabaseProvider.PROP_DATABASE, db);
		
		ServiceChecker<MongoDatabaseProvider> dbChecker = createCheckerTrackedForCleanUp(MongoDatabaseProvider.class);
		dbChecker.start();
		Configuration dbConfig = createConfigForCleanup(ConfigurationProperties.DATABASE_PID, "?", dbp);
		assertTrue(dbChecker.awaitCreation());
		
		Dictionary<String, Object> properties = new Hashtable<>();
		
		String repoId = "test_repo";
		properties.put(EMFRepository.PROP_ID, repoId);
		properties.put(EMFRepository.PROP_BASE_URI, clientUri + "/" + db);
		
		ServiceChecker<EMFRepository> repoChecker = createCheckerTrackedForCleanUp(EMFRepository.class);
		repoChecker.start();
		
		createConfigForCleanup(EMFMongoConfiguratorConstants.SINGLETON_REPOSITORY_CONFIGURATION_NAME, "?", properties);
		
		assertTrue(repoChecker.awaitCreation());
		
		
		MongoDatabase database = client.getDatabase(db);
		collection = database.getCollection("Person"); 
		collection.drop();
		
		CountDownLatch latch = new CountDownLatch(1);
		latch.await(1, TimeUnit.SECONDS);
		
		EMFRepository repository = getService(FrameworkUtil.createFilter("(" + EMFRepository.PROP_ID + "=" + repoId + ")"), 5000);
		assertNotNull(repository);
		
		Address address = TestFactory.eINSTANCE.createAddress();
		address.setId("testAddress");
		
		Person person = TestFactory.eINSTANCE.createPerson();
		person.setId("test");
		person.setFirstName("Emil");
		person.setLastName("Tester");
		person.setAddress(address);
		URI uri = repository.createUri(person);
		assertEquals(clientUri + "/" + db + "/Person/test", uri.toString());
		assertEquals(0, collection.countDocuments());
		
		repository.save(person);
		
		assertEquals(1, collection.countDocuments());
		
		Resource r = person.eResource();
		assertNotNull(r);
		ResourceSet rs = r.getResourceSet();
		assertNotNull(rs);
		assertEquals(1, rs.getResources().size());
		
		repository.detach(person);
		assertNull(person.eResource());
		assertEquals(0, rs.getResources().size());
		
		Person personResult = repository.getEObject(TestPackage.Literals.PERSON, "test");
		assertNotNull(personResult);
		assertNotEquals(person, personResult);
		assertNotEquals(r, personResult.eResource());
		
		assertFalse(EcoreUtil.equals(person, personResult));
		assertNotNull(personResult.getAddress());
		assertTrue(personResult.getAddress().eIsProxy());
		
		repository.detach(personResult);
		
		repository.save(person, address);
		
		repository.detach(person);
		repository.detach(address);
		
		personResult = repository.getEObject(TestPackage.Literals.PERSON, "test");
		assertNotNull(personResult);
		assertNotEquals(person, personResult);
		assertNotEquals(r, personResult.eResource());
		
		assertTrue(EcoreUtil.equals(person, personResult));
		assertNotNull(personResult.getAddress());
		assertFalse(personResult.getAddress().eIsProxy());
		
		assertNotEquals(address, personResult.getAddress());
		
		deleteConfigurationAndRemoveFromCleanup(dbConfig);
	}
}
