/**
 * Copyright (c) 2014 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.emf.repository.mongo.impl;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.gecko.collection.EReferenceCollection;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.emf.repository.DefaultEMFRepository;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.mongo.query.MongoQueryBuilder;
import org.gecko.emf.repository.query.IQuery;
import org.gecko.emf.repository.query.IQueryBuilder;
import org.gecko.emf.repository.query.QueryRepository;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;
import org.osgi.service.component.annotations.ServiceScope;
import org.gecko.emf.repository.mongo.api.EMFMongoConfiguratorConstants;

/**
 * EMF repository to store emf objects into a mongo database.
 * @author Mark Hoffmann
 * @since 12.04.2015
 */
@Component(
		name=EMFMongoConfiguratorConstants.PROTOTYPE_REPOSITORY_CONFIGURATION_NAME, 
		service=EMFRepository.class, 
		configurationPid=EMFMongoConfiguratorConstants.PROTOTYPE_REPOSITORY_CONFIGURATION_NAME, 
		configurationPolicy=ConfigurationPolicy.REQUIRE, 
		scope = ServiceScope.PROTOTYPE,
		property= {"persistence=mongo"}
		)
public class PrototypeEMFMongoRepository extends DefaultEMFRepository implements QueryRepository {

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#activate(java.util.Map)
	 */
	@Override
	@Activate
	public void activate(Map<String, ?> properties) {
		super.activate(properties);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#deactivate()
	 */
	@Override
	@Deactivate
	public void deactivate() {
		super.deactivate();
	}
	
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#setResourceSetFactory(org.gecko.emf.osgi.ResourceSetFactory)
	 */
	@Override
	@Reference(name="ResourceSetFactory", cardinality=ReferenceCardinality.MANDATORY, policy=ReferencePolicy.STATIC, unbind="unsetResourceSetFactory")
	public void setResourceSetFactory(ResourceSetFactory resourceSetFactory) {
		super.setResourceSetFactory(resourceSetFactory);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#createResource(org.eclipse.emf.ecore.EObject, java.lang.String)
	 */
	@Override
	public synchronized Resource createResource(EObject object, String contentType) {
		if (object == null) {
			return null;
		}
		Resource resultResource = object.eResource();
		if (resultResource != null && resultResource.getResourceSet() != null && "mongodb".equals(resultResource.getURI().scheme())) {
			resultResource = object.eResource();
		} else {
			URI uri = createUri(object);
			resultResource = getResourceSet().getResource(uri, false);
			if (resultResource == null) {
				resultResource = getResourceSet().createResource(uri, contentType);
			}
		}
		return resultResource;
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#getAllEObjects(org.eclipse.emf.ecore.EClass, java.util.Map)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <T extends EObject> List<T> getAllEObjects(EClass eClass, Map<?, ?> options) {
		IQuery query = createQueryBuilder().allQuery().build();
		return getEObjectsByQuery(eClass, query, (Map<Object, Object>) options);
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#getAllEObjects(org.eclipse.emf.ecore.EClass)
	 */
	@Override
	public List<EObject> getAllEObjects(EClass eClass) {
		return getAllEObjects(eClass, Collections.emptyMap());
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.query.QueryRepository#createQueryBuilder()
	 */
	@Override
	public IQueryBuilder createQueryBuilder() {
		return new MongoQueryBuilder();
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.query.QueryRepository#getEObjectsByQuery(org.eclipse.emf.ecore.EClass, org.gecko.emf.repository.query.IQuery)
	 */
	@Override
	public <T extends EObject> List<T> getEObjectsByQuery(EClass eClass, IQuery query) {
		return getEObjectsByQuery(eClass, query, defaultResourceSetLoadOptions);
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.query.QueryRepository#getEObjectsByQuery(org.eclipse.emf.ecore.EClass, org.gecko.emf.repository.query.IQuery, java.util.Map)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <T extends EObject> List<T> getEObjectsByQuery(EClass eClass, IQuery query,
			Map<Object, Object> options) {
		Resource resource = executeQuery(eClass, query, options);
		if (resource.getContents().size() > 0) {
			EReferenceCollection result = (EReferenceCollection) resource.getContents().get(0);
			return (List<T>) result.getValues();
		} else {
			return Collections.emptyList();
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.query.QueryRepository#getEObjectByQuery(org.eclipse.emf.ecore.EClass, org.gecko.emf.repository.query.IQuery, java.util.Map)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <T extends EObject> T getEObjectByQuery(EClass eClass, IQuery query, Map<Object, Object> options) {
		Resource resource = executeQuery(eClass, query, options);
		if(resource.getContents().isEmpty()){
			return null;
		} else {
			T result = null;
			if(resource.getContents().get(0) instanceof EReferenceCollection){
				EReferenceCollection referenceCollection = (EReferenceCollection) resource.getContents().get(0);
				if(referenceCollection.getValues().size() > 0){
					result = (T) referenceCollection.getValues().get(0);
				}
				resource.getContents().clear();
				resource.getResourceSet().getResources().remove(resource);
			} else {
				result =  (T) resource.getContents().get(0);
			}
			return result;
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.EMFRepository#getAdapter(java.lang.Class)
	 */
	@Override
	public Object getAdapter(Class<?> adapter) {
		if (adapter.isAssignableFrom(QueryRepository.class)) {
			return this;
		}
		if (adapter.isAssignableFrom(IQueryBuilder.class)) {
			return createQueryBuilder();
		}
		return null;
	}

	/**
	 * Executes the fiven query and returns the result as a resource
	 * @param eClass the {@link EClass} to look for
	 * @param query the query
	 * @param options the options {@link Map}
	 * @return a {@link Resource} containing the desired result (or is empty)
	 */
	private Resource executeQuery(EClass eClass, IQuery query,
			Map<Object, Object> options) {
		if (eClass == null || query == null) {
			throw new IllegalStateException("Cannot get EObjects by query because of missing parameters: EClass: " + eClass + ", IQuery: " + query);
		}
		String queryString = MongoQueryBuilder.createMongoQuery(query);
		if (queryString != null && queryString.length() < 3) {
			queryString = "{ filter: " + queryString + " }";
		}
		URI uri = createEClassUri(eClass);
		uri = URI.createURI(uri.toString() + "/?" + queryString);
		org.eclipse.emf.ecore.resource.Resource resource = createResource(uri);
		try {
			resource.load(options);
		} catch (IOException e) {
			throw new IllegalStateException("Could not load EObjects by query with message " + e.getMessage(), e);
		}
		return resource;
	}
}
