/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.rsa.core;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.emf.ecore.xml.namespace.XMLNamespacePackage;
import org.eclipse.emf.ecore.xml.type.XMLTypePackage;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.osgi.util.promise.Deferred;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;

/**
 * Serializes and De-serializes the end-point descriptions
 * @author Mark Hoffmann
 * @since 05.07.2018
 */
public abstract class EObjectDeAndSerializer<D extends EObject, S extends EObject> implements DeSerializer<D, DeSerializationContext>, Serializer<S, SerializationContext>{

	private static final Logger logger = Logger.getLogger(EObjectDeAndSerializer.class.getName());
	private final ResourceSet resourceSet;
	private final Map<String, Object> binaryOptions = new HashMap<String, Object>();
	private final DeSerializationContext deserContext;
	private final SerializationContext serContext;

	/**
	 * Creates a new instance.
	 * @param dsContext de-serialization context
	 * @param sContext serialization context
	 */
	public EObjectDeAndSerializer(DeSerializationContext dsContext, SerializationContext sContext) {
		this.serContext = sContext;
		this.deserContext = dsContext;
		resourceSet = new ResourceSetImpl();
		configureResourceSet();
	}
	
	/**
	 * Creates a new instance.
	 * @param rsf the resource set factory
	 * @param dsContext de-serialization context
	 * @param sContext serialization context
	 */
	public EObjectDeAndSerializer(ResourceSetFactory rsf, DeSerializationContext dsContext, SerializationContext sContext) {
		this.serContext = sContext;
		this.deserContext = dsContext;
		resourceSet = rsf.createResourceSet();
		configureResourceSet();
	}

	/**
	 * Returns the {@link ResourceSet}
	 * @return the {@link ResourceSet}
	 */
	public ResourceSet getResourceSet() {
		return resourceSet;
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.rsa.core.DeSerializer#deserialize(java.io.InputStream)
	 */
	@Override
	public Promise<D> deserialize(InputStream input) {
		return deserialize(input, getDeSerializationContext());
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.rsa.core.DeSerializer#deserialize(java.io.InputStream, org.gecko.rsa.core.DeSerializationContext)
	 */
	@Override
	public Promise<D> deserialize(InputStream input, DeSerializationContext context) {
		Deferred<D> result = new Deferred<D>();
		PromiseFactory.inlineExecutor().execute(()->{
			synchronized (resourceSet) {
				try {
					Resource resource = resourceSet.createResource(URI.createURI("tmp.xml"));
					resource.load(input, context.getDeSerializationOptions());
					if (!resource.getContents().isEmpty()) {
						EObject root = resource.getContents().get(0);
						resource.getContents().clear();
						resourceSet.getResources().remove(resource);
						D returnObject = getDeSerializationContent(root);
						result.resolve(returnObject);
					} else {
						logger.severe("Cannot de-serialize the payload into an EObject. Resources' content is empty");
						result.resolve(null);
					}
				} catch (IOException e) {
					logger.log(Level.SEVERE, "Error de-serializing binary data into an EObject because of an IO error", e);
					result.fail(e);
				} catch (Exception e) {
					logger.log(Level.SEVERE, "Error de-serializing binary data into an EObject", e);
					result.fail(e);
				}
			}
		});
		return result.getPromise();
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.rsa.core.Serializer#serialize(java.lang.Object)
	 */
	@Override
	public Promise<OutputStream> serialize(S object) {
		return serialize(object, getSerializationContext());
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.rsa.core.Serializer#serialize(java.lang.Object, org.gecko.rsa.core.SerializationContext)
	 */
	@Override
	public Promise<OutputStream> serialize(S object, SerializationContext context) {
		return serialize(object, null, context);
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.rsa.core.Serializer#serialize(java.lang.Object, java.io.OutputStream)
	 */
	@Override
	public Promise<OutputStream> serialize(S object, OutputStream output) {
		return serialize(object, output, getSerializationContext());
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.rsa.core.Serializer#serialize(java.lang.Object, java.io.OutputStream, org.gecko.rsa.core.SerializationContext)
	 */
	@Override
	public Promise<OutputStream> serialize(S descriptions, OutputStream output, SerializationContext context) {
		Deferred<OutputStream> result = new Deferred<OutputStream>();
		PromiseFactory.inlineExecutor().execute(()->{
			OutputStream thisOutput = output == null ? new ByteArrayOutputStream() : output;
			EObject root = getSerializationContent(descriptions);
			synchronized (resourceSet) {
				Resource resource = resourceSet.createResource(URI.createURI("tmp.xml"));
				resource.getContents().add(root);
				try {
					resource.save(thisOutput, context.getSerializationOptions());
					resource.getContents().clear();
					resourceSet.getResources().remove(resource);
					result.resolve(thisOutput);
				} catch (IOException e) {
					logger.log(Level.SEVERE, "Error serializing binary data, because of an IO error", e);
					result.fail(e);
				}
			}
		});
		return result.getPromise();
	}
	
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.rsa.core.Serializer#getSerializationContext()
	 */
	@Override
	public SerializationContext getSerializationContext() {
		return serContext;
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.rsa.core.DeSerializer#getDeSerializationContext()
	 */
	@Override
	public DeSerializationContext getDeSerializationContext() {
		return deserContext;
	}

	/**
	 * Configures the resource set 
	 */
	private void configureResourceSet() {
		XMLTypePackage.eINSTANCE.eClass();
		XMLNamespacePackage.eINSTANCE.eClass();
		EcorePackage.eINSTANCE.eClass();
		// register all model packages to the registry
		EPackage xmlTypePackage = XMLTypePackage.eINSTANCE;
		EPackage xmlNamespacePackage = XMLNamespacePackage.eINSTANCE;
		EPackage ecorePackage = EcorePackage.eINSTANCE;
		resourceSet.getPackageRegistry().put(EcorePackage.eNS_URI, ecorePackage);
		resourceSet.getPackageRegistry().put(XMLTypePackage.eNS_URI, xmlTypePackage);
		resourceSet.getPackageRegistry().put(XMLNamespacePackage.eNS_URI, xmlNamespacePackage);
		binaryOptions.put(XMLResource.OPTION_BINARY, Boolean.TRUE);
		doConfigureResourceSet(resourceSet);
	}

	/**
	 * Clients may implement own resource set configuration
	 * @param resourceSet the {@link ResourceSet} to be configured
	 */
	protected abstract void doConfigureResourceSet(ResourceSet resourceSet);

	/**
	 * Returns the final {@link EObject} that can be serialized. Additional operations can be executed here, to
	 * get the final serialization {@link EObject}.
	 * @param serObject the object to serialized
	 * @return the final object that will be serialized
	 */
	protected abstract EObject getSerializationContent(S serObject);
	
	/**
	 * Returns the final {@link D} from the resource content. Additional operations can be executed here, to
	 * get the final de-serialization object.
	 * @param content the content object that was de-serialized
	 * @return the final object
	 */
	protected abstract D getDeSerializationContent(EObject content);

}
