/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.gecko.rsa.core.converter;

import java.io.StringWriter;
import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.util.FeatureMap;
import org.eclipse.emf.ecore.xml.type.XMLTypePackage;
import org.gecko.rsa.model.rsa.Property;
import org.gecko.rsa.model.rsa.RSAFactory;
import org.gecko.rsa.model.rsa.RSAPackage;
import org.gecko.rsa.model.rsa.Value;
import org.gecko.rsa.model.rsa.XmlType;
import org.w3c.dom.Node;

/**
 * Mapper from {@link Map} into the serialization model and back
 * @author Mark Hoffmann
 * @since 06.07.2018
 */
public class PropertiesMapper {
	
    private static final Logger logger = Logger.getLogger(PropertiesMapper.class.getName());

    public Map<String, Object> toProps(List<Property> properties) {
        Map<String, Object> map = new HashMap<String, Object>();
        for (Property prop : properties) {
            map.put(prop.getName(), getValue(prop));
        }
        return map;
    }

    private Object getValue(Property prop) {
        String type = getTypeName(prop);
        if (prop.eIsSet(RSAPackage.Literals.PROPERTY__ARRAY)) {
        	return getArray(prop.getArray(), type);
        }
        if (prop.eIsSet(RSAPackage.Literals.PROPERTY__LIST)) {
        	return handleCollection(prop.getList(), new ArrayList<Object>(), type);
        }
        if (prop.eIsSet(RSAPackage.Literals.PROPERTY__SET)) {
        	return handleCollection(prop.getList(), new HashSet<Object>(), type);
        }
        if (prop.eIsSet(RSAPackage.Literals.PROPERTY__VALUE)) {
        	return instantiate(type, prop.getValue());
        }
        if (prop.eIsSet(RSAPackage.Literals.PROPERTY__XML)) {
        	return readXML(prop.getXml(), type);
        }
        if (prop.eIsSet(RSAPackage.Literals.PROPERTY__MIXED)) {
        	FeatureMap mixed = prop.getMixed();
        	EList<Object> list = mixed.list(XMLTypePackage.Literals.XML_TYPE_DOCUMENT_ROOT__TEXT);
        	if (!list.isEmpty()) {
        		return instantiate(type, list.get(0).toString());
        	}
        }
        return null;
    }

    private static String getTypeName(Property prop) {
        String type = prop.getValueType();
        return type == null ? "String" : type;
    }

    private Object getArray(org.gecko.rsa.model.rsa.Array arrayEl, String type) {
        List<Value> values = arrayEl.getValue();
        Class<?> cls = null;
        if ("long".equals(type)) {
            cls = long.class;
        } else if ("double".equals(type)) {
            cls = double.class;
        } else if ("float".equals(type)) {
            cls = float.class;
        } else if ("int".equals(type)) {
            cls = int.class;
        } else if ("byte".equals(type)) {
            cls = byte.class;
        } else if ("boolean".equals(type)) {
            cls = boolean.class;
        } else if ("short".equals(type)) {
            cls = short.class;
        }

        try {
            if (cls == null) {
                cls = ClassLoader.getSystemClassLoader().loadClass("java.lang." + type);
            }
            Object array = Array.newInstance(cls, values.size());

            for (int i = 0; i < values.size(); i++) {
                Object val = getValue(values.get(i), type);
                Array.set(array, i, val);
            }

            return array;
        } catch (Exception e) {
            logger.log(Level.WARNING, "Could not create array for Endpoint Description", e);
            return null;
        }
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    private Collection handleCollection(org.gecko.rsa.model.rsa.Array el, Collection value, String type) {
        List<Value> values = el.getValue();
        for (Value val : values) {
            Object obj = getValue(val, type);
            value.add(obj);
        }
        return value;
    }
    
    private Object getValue(Value value, String type) {
    	EList<Object> list = value.getMixed().list(XMLTypePackage.Literals.XML_TYPE_DOCUMENT_ROOT__TEXT);
        if (list.size() == 1 && list.get(0) instanceof String) {
            return handleValue((String)list.get(0), type);
        }
        return "";
    }

    private String readXML(XmlType el, String type) {
        if (el == null) {
            return null;
        }
        if (!"String".equals(type)) {
            logger.warning("Embedded XML must be of type String, found: " + type);
            return null;
        }
        Node xmlContent = (Node)el.getAny();
        xmlContent.normalize();
        try {
            TransformerFactory transFactory = TransformerFactory.newInstance();
            Transformer transformer = transFactory.newTransformer();
            StringWriter buffer = new StringWriter();
            transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
            transformer.transform(new DOMSource(xmlContent), new StreamResult(buffer));
            return buffer.toString();
        } catch (Exception e) {
            return "";
        }
    }

    private static Object handleValue(String val, String type) {
        return instantiate(type, val);
    }

    private static Object instantiate(String type, String value) {
        if ("String".equals(type)) {
            return value;
        }

        value = value.trim();
        String boxedType = null;
        if ("long".equals(type)) {
            boxedType = "Long";
        } else if ("double".equals(type)) {
            boxedType = "Double";
        } else if ("float".equals(type)) {
            boxedType = "Float";
        } else if ("int".equals(type)) {
            boxedType = "Integer";
        } else if ("byte".equals(type)) {
            boxedType = "Byte";
        } else if ("char".equals(type)) {
            boxedType = "Character";
        } else if ("boolean".equals(type)) {
            boxedType = "Boolean";
        } else if ("short".equals(type)) {
            boxedType = "Short";
        }

        if (boxedType == null) {
            boxedType = type;
        }
        String javaType = "java.lang." + boxedType;

        try {
            if ("Character".equals(boxedType)) {
                return new Character(value.charAt(0));
            } else {
                Class<?> cls = ClassLoader.getSystemClassLoader().loadClass(javaType);
                Constructor<?> ctor = cls.getConstructor(String.class);
                return ctor.newInstance(value);
            }
        } catch (Exception e) {
            logger.warning("Could not create Endpoint Property of type " + type + " and value " + value);
            return null;
        }
    }
    
    public List<Property> fromProps(Map<String, Object> m) {
        List<Property> props = new ArrayList<Property>();
        for (Map.Entry<String, Object> entry : m.entrySet()) {
            String key = entry.getKey();
            Object val = entry.getValue();

            Property propEl = RSAFactory.eINSTANCE.createProperty();
            propEl.setName(key);
            if (val.getClass().isArray()) {
                org.gecko.rsa.model.rsa.Array arrayEl = RSAFactory.eINSTANCE.createArray();
                propEl.setArray(arrayEl);
                for (Object o : normalizeArray(val)) {
                    setValueType(propEl, o);
                    Value v = RSAFactory.eINSTANCE.createValue();
                    arrayEl.getValue().add(v);
                    v.getMixed().list(XMLTypePackage.Literals.XML_TYPE_DOCUMENT_ROOT__TEXT).add(o.toString());
                }
            } else if (val instanceof List) {
            	org.gecko.rsa.model.rsa.Array listEl = RSAFactory.eINSTANCE.createArray();
            	propEl.setList(listEl);
                handleCollectionValue((Collection<?>) val, propEl, listEl);
            } else if (val instanceof Set) {
            	org.gecko.rsa.model.rsa.Array setEl = RSAFactory.eINSTANCE.createArray();
            	propEl.setSet(setEl);
                handleCollectionValue((Collection<?>) val, propEl, setEl);
            } else if (val instanceof String
                    || val instanceof Character
                    || val instanceof Boolean
                    || val instanceof Byte) {
                setValueType(propEl, val);
                propEl.setValue(val.toString());
            } else if (val instanceof Long
                    || val instanceof Double
                    || val instanceof Float
                    || val instanceof Integer
                    || val instanceof Short) {
                // various numbers..   maybe "val instanceof Number"?
                setValueType(propEl, val);
                propEl.setValue(val.toString());
            } else {
                // Don't add this property as the value type is not supported
                continue;
            }
            props.add(propEl);
        }
        return props;
    }

    private static Object[] normalizeArray(Object val) {
        List<Object> l = new ArrayList<Object>();
        if (val instanceof int[]) {
            int[] ia = (int[]) val;
            for (int i : ia) {
                l.add(i);
            }
        } else if (val instanceof long[]) {
            long[] la = (long[]) val;
            for (long i : la) {
                l.add(i);
            }
        } else if (val instanceof float[]) {
            float[] fa = (float[]) val;
            for (float f : fa) {
                l.add(f);
            }
        } else if (val instanceof byte[]) {
            byte[] ba = (byte[]) val;
            for (byte b : ba) {
                l.add(b);
            }
        } else if (val instanceof boolean[]) {
            boolean[] ba = (boolean[]) val;
            for (boolean b : ba) {
                l.add(b);
            }
        } else if (val instanceof short[]) {
            short[] sa = (short[]) val;
            for (short s : sa) {
                l.add(s);
            }
        } else if (val instanceof char[]) {
            char[] ca = (char[]) val;
            for (char c : ca) {
                l.add(c);
            }
        } else {
            return (Object[]) val;
        }
        return l.toArray();
    }

    private static void handleCollectionValue(Collection<?> val, Property propEl, org.gecko.rsa.model.rsa.Array listEl) {
        for (Object o : val) {
            setValueType(propEl, o);
            propEl.getMixed().list(XMLTypePackage.Literals.XML_TYPE_DOCUMENT_ROOT__TEXT).add(o.toString());
        }
    }

    private static void setValueType(Property propEl, Object val) {
        if (val instanceof String) {
            return;
        }

        String dataType = val.getClass().getName();
        if (dataType.startsWith("java.lang.")) {
            dataType = dataType.substring("java.lang.".length());
        }
        propEl.setValueType(dataType);
    }
}
