/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.rsa.provider;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.apache.aries.rsa.spi.Endpoint;
import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.osgi.messaging.Message;
import org.gecko.osgi.messaging.MessagingService;
import org.gecko.rsa.core.DeSerializationContext;
import org.gecko.rsa.core.DeSerializer;
import org.gecko.rsa.core.SerializationContext;
import org.gecko.rsa.core.Serializer;
import org.gecko.rsa.core.converter.PropertiesMapper;
import org.gecko.rsa.model.rsa.Property;
import org.gecko.rsa.provider.ser.BasicObjectInputStream;
import org.gecko.rsa.provider.ser.BasicObjectOutputStream;
import org.gecko.rsa.provider.ser.RequestDeSerializer;
import org.gecko.rsa.provider.ser.ResponseDeSerializer;
import org.gecko.rsa.rsaprovider.EObjectRequestParameter;
import org.gecko.rsa.rsaprovider.RSAProviderFactory;
import org.gecko.rsa.rsaprovider.RSARequest;
import org.gecko.rsa.rsaprovider.RSAResponse;
import org.gecko.rsa.rsaprovider.RequestParameter;
import org.gecko.rsa.rsaprovider.ResponseCodeType;
import org.osgi.service.remoteserviceadmin.EndpointDescription;
import org.osgi.service.remoteserviceadmin.RemoteConstants;
import org.osgi.util.promise.Promise;
import org.osgi.util.pushstream.PushStream;

/**
 * An messaging adapter based aries endpoint
 * @author Mark Hoffmann
 * @since 07.07.2018
 */
public class MessagingRSAEndpoint implements Endpoint {
	
	public static final String MA_DATA_TOPIC = "gecko/rsa/data/%s";
	public static final String MA_DATA_RESPONSE_TOPIC = "gecko/rsa/data/%s/response";
	private static final Logger logger = Logger.getLogger(MessagingRSAEndpoint.class.getName());
	private final MessagingService messaging;
	private final PropertiesMapper mapper = new PropertiesMapper();
	private EndpointDescription epd;
	private PushStream<Message> receiveData = null;
	private String requestResponseAddress;
	private final ClassLoader serviceCL;
	private MethodInvoker invoker;
	private final DeSerializer<RSARequest, DeSerializationContext> deserializer;
	private final Serializer<RSAResponse, SerializationContext> serializer;

	/**
	 * Creates a new instance.
	 */
	public MessagingRSAEndpoint(MessagingService messaging, ResourceSetFactory resourceSetFactory, Object service, Map<String, Object> effectiveProperties) {
		this.messaging = messaging;
		deserializer = new RequestDeSerializer(resourceSetFactory);
		serializer = new ResponseDeSerializer(resourceSetFactory);
		if (service == null) {
			throw new NullPointerException("Service must not be null");
		}
		this.serviceCL = service.getClass().getClassLoader();
		this.invoker = new MethodInvoker(service);
        if (effectiveProperties.get(MessagingRSAProvider.MA_CONFIG_TYPE + ".id") != null) {
            throw new IllegalArgumentException("For the tck .. Just to please you!");
        }
        List<Property> props = mapper.fromProps(effectiveProperties);
        Optional<String> endpointIdOpt = props.stream().filter(p->p.getName().equals(MessagingRSAProvider.GECKO_RSA_ID)).map(p->p.getValue()).findFirst();
        String endpointId = endpointIdOpt.orElse(null);
        if (endpointId == null) {
        	throw new IllegalArgumentException("Remote constant Endpoint_Id is missing");
        }
        initializeMessaging(endpointId);
        effectiveProperties.put(RemoteConstants.ENDPOINT_ID, endpointId);
        effectiveProperties.put(RemoteConstants.SERVICE_EXPORTED_CONFIGS, "");
        effectiveProperties.put(RemoteConstants.SERVICE_INTENTS, Arrays.asList("osgi.basic", "osgi.async"));
        
        // tck tests for one such property ... so we provide it
        effectiveProperties.put(MessagingRSAProvider.MA_CONFIG_TYPE + ".id", endpointId);
        this.epd = new EndpointDescription(effectiveProperties);
	}
	
	/**
	 * Initializes the messaging system
	 * @param endpointId the endpointId
	 */
	private void initializeMessaging(String endpointId) {
		this.requestResponseAddress = String.format(MessagingRSAEndpoint.MA_DATA_RESPONSE_TOPIC, endpointId);
		String requestDataAddress = String.format(MessagingRSAEndpoint.MA_DATA_TOPIC, endpointId);
		try {
			this.receiveData = messaging.subscribe(requestDataAddress);
//			this.receiveData.forEach(this::handleRequest);
			this.receiveData.forEach(this::handleRequestNew);
		} catch (Exception e) {
			logger.log(Level.SEVERE, String.format("Error subscribing to receiver topic '%s'", endpointId));
		}
		
	}
	
	/**
	 * Handles the message request
	 * @param message the message object with the payload
	 */
	@SuppressWarnings("unused")
	private void handleRequest(Message message) {
		ByteBuffer buffer = message.payload();
		ByteArrayInputStream input = new ByteArrayInputStream(buffer.array());
		ByteArrayOutputStream output = new ByteArrayOutputStream();
		try (ObjectInputStream objectInput = new BasicObjectInputStream(input, serviceCL);
				ObjectOutputStream objectOutput = new BasicObjectOutputStream(output)) {
			String id = (String) objectInput.readObject();
			objectOutput.writeObject(id);
			handleCall(objectInput, objectOutput);
			ByteBuffer responseData = ByteBuffer.wrap(output.toByteArray());
			messaging.publish(requestResponseAddress, responseData);
		} catch (IOException e) {
			logger.log(Level.SEVERE, "Cannot create BasicInputStream from byte array", e);
		} catch (ClassNotFoundException e) {
			logger.log(Level.SEVERE, "Cannot find class to read UUID", e);
		} catch (Exception e) {
			logger.log(Level.SEVERE, "Cannot handle request because of an error", e);
		}
	}
	
	/**
	 * Handles the message request
	 * @param message the message object with the payload
	 */
	private void handleRequestNew(Message message) {
		ByteBuffer buffer = message.payload();
		ByteArrayInputStream input = new ByteArrayInputStream(buffer.array());
		try {
			Promise<RSARequest> requestPromise = deserializer.deserialize(input);
			RSARequest request = requestPromise.getValue();
			RSAResponse response = handleResponse(request);
			if (response == null) {
				throw new IllegalStateException("Cannot handle the request " + request);
			}
			Promise<OutputStream> responsePromise = serializer.serialize(response);
			ByteArrayOutputStream responseOS = responsePromise.filter(os->os instanceof ByteArrayOutputStream).map(os->(ByteArrayOutputStream)os).getValue();
			if (responseOS != null) {
				byte[] payload = responseOS.toByteArray();
				ByteBuffer responseData = ByteBuffer.wrap(payload);
				messaging.publish(requestResponseAddress, responseData);
				
			}
		} catch (Exception e) {
			logger.log(Level.SEVERE, "Cannot handle request because of an error", e);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see java.io.Closeable#close()
	 */
	@Override
	public void close() throws IOException {
		if (receiveData != null) {
			receiveData.close();
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see org.apache.aries.rsa.spi.Endpoint#description()
	 */
	@Override
	public EndpointDescription description() {
		return epd;
	}
	
	/**
	 * Handles the request call
	 * @param requestInput the request input
	 * @param objectreponseOutput the request response data
	 * @throws Exception
	 */
	private void handleCall(ObjectInputStream requestInput, ObjectOutputStream objectreponseOutput) throws Exception {
        String methodName = (String)requestInput.readObject();
        Object[] args = (Object[])requestInput.readObject();
        Object result = invoker.invoke(methodName, args);
        result = resolveAsnyc(result);
        if (result instanceof InvocationTargetException) {
            result = ((InvocationTargetException) result).getCause();
        }
        objectreponseOutput.writeObject(result);
    }

    /**
	 * @param request
	 * @return
	 */
	private RSAResponse handleResponse(RSARequest request) {
		if (request == null) {
			return null;
		}
		String methodName = request.getServiceName();
		Set<Object> parameter = request.getParameter().
				stream().
				sorted((p1,p2)->Integer.valueOf(p1.getNumber()).compareTo(Integer.valueOf(p2.getNumber()))).
				map(this::mapParameter).
				collect(Collectors.toCollection(LinkedHashSet::new));
        Object result = invoker.invoke(methodName, parameter.toArray(new Object[parameter.size()]));
        RSAResponse response = RSAProviderFactory.eINSTANCE.createRSAResponse();
        response.setId(request.getId());
        response.setCode(ResponseCodeType.OK);
        if (result instanceof EObject) {
        	response.setEObjectResult(true);
        	response.setEObject((EObject) result);
        } else {
        	response.setObject(result);
        }
//        response.setRequest(request);
		return response;
	}

	/**
     * If there is a async request, we have to handle it this way
     * @param result the result object
     * @return the async result
     * @throws InterruptedException
     */
    @SuppressWarnings("unchecked")
    private Object resolveAsnyc(Object result) throws InterruptedException {
        if (result instanceof Future) {
            Future<Object> fu = (Future<Object>) result;
            try {
                result = fu.get();
            } catch (ExecutionException e) {
                result = e.getCause();
            }
        } else if (result instanceof CompletionStage) {
            CompletionStage<Object> fu = (CompletionStage<Object>) result;
            try {
                result = fu.toCompletableFuture().get();
            } catch (ExecutionException e) {
                result = e.getCause();
            }
        } else if (result instanceof Promise) {
            Promise<Object> fu = (Promise<Object>) result;  
            try {
                result = fu.getValue();
            } catch (InvocationTargetException e) {
                result = e.getCause();
            }
        }
        return result;
    }
    
    private Object mapParameter(RequestParameter parameter) {
    	if (parameter instanceof EObjectRequestParameter) {
			return ((EObjectRequestParameter)parameter).getEObject();
		} else {
			return parameter.getObject();
		}
    }

}
