/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.gecko.rsa.provider.ser;

import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectStreamClass;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.osgi.framework.Version;

public class BasicObjectInputStream extends ObjectInputStream {
	
	private static final Logger logger = Logger.getLogger(BasicObjectInputStream.class.getName());

    private ClassLoader loader;

    public BasicObjectInputStream(InputStream in, ClassLoader loader) throws IOException {
        super(in);
        AccessController.doPrivileged(new PrivilegedAction<Void>() {
            public Void run() {
                enableResolveObject(true);
                return null;
            }
        });
        this.loader = loader;
    }

    @Override
    protected Class<?> resolveClass(ObjectStreamClass desc) throws IOException, ClassNotFoundException {
        try {
            String className = desc.getName();
            // Must use Class.forName instead of loader.loadClass to handle cases like array of user classes
            return Class.forName(className, false, loader);
        } catch (ClassNotFoundException e) {
            logger.log(Level.SEVERE, "Error loading class using classloader of user bundle. Trying our own ClassLoader now", e);
            return super.resolveClass(desc);
        }
    }
    
    @Override
    protected Object resolveObject(Object obj) throws IOException {
        if (obj instanceof VersionMarker) {
            VersionMarker verionMarker = (VersionMarker)obj;
            return Version.parseVersion(verionMarker.getVersion());
        } else if (obj instanceof DTOMarker) {
            DTOMarker dtoMarker = (DTOMarker)obj;
            return dtoMarker.getDTO(loader);
        } else {
            return super.resolveObject(obj);
        }
    }
}
