/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.runtime.application.internal;

import java.security.AccessController;
import java.security.PrivilegedAction;

import org.osgi.framework.BundleContext;
import org.osgi.framework.Filter;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;

/**
 * Service tracker that checks operations using the {@link SecurityManager}
 * @author Mark Hoffmann
 * @since 23.03.2018
 */
public class SecureServiceTracker<S,T> {

	private final ServiceTracker<S, T> tracker;

	/**
	 * Creates a new instance.
	 */
	public SecureServiceTracker(BundleContext context, String filter, ServiceTrackerCustomizer<S, T> customizer) {
		tracker = new ServiceTracker<S, T>(context, filter, customizer);
	}

	/**
	 * Creates a new instance.
	 */
	public SecureServiceTracker(BundleContext context, String filter) {
		tracker = new ServiceTracker<S, T>(context, filter, null);
	}

	/**
	 * Creates a new instance.
	 */
	public SecureServiceTracker(BundleContext context, Filter filter, ServiceTrackerCustomizer<S, T> customizer) {
		tracker = new ServiceTracker<S, T>(context, filter, customizer);
	}

	/**
	 * Creates a new instance.
	 */
	public SecureServiceTracker(BundleContext context, Filter filter) {
		tracker = new ServiceTracker<S, T>(context, filter, null);
	}

	/**
	 * Creates a new instance.
	 */
	public SecureServiceTracker(BundleContext context, Class<S> clazz, ServiceTrackerCustomizer<S, T> customizer) {
		tracker = new ServiceTracker<S, T>(context, clazz, customizer);
	}

	/**
	 * Opens the service tracker
	 * @param allServices <code>true</code> to track all services
	 */
	public void open(final boolean allServices) {
		if (System.getSecurityManager() == null) {
			tracker.open(allServices);
		} else {
			AccessController.doPrivileged(new PrivilegedAction<T>() {
				public T run() {
					tracker.open(allServices);
					return null;
				}
			});
		}
	}
	
	/**
	 * Closes the service tracker
	 */
	public void close() {
		tracker.close();
	}

	/**
	 * Returns the service from the given service tracker
	 * @return the service instances
	 */
	public T getService() {
		if (System.getSecurityManager() == null)
			return tracker.getService();
		return AccessController.doPrivileged(new PrivilegedAction<T>() {
			public T run() {
				return tracker.getService();
			}
		});
	}

}
