/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.runtime.application.helper;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import org.osgi.service.application.ApplicationException;

/**
 * Helper class to check properties and arguments
 * @author Mark Hoffmann
 * @since 23.03.2018
 */
public class ArgumentsHelper {
	
	@SuppressWarnings("rawtypes")
	private static final Collection scalars = Arrays.asList(new Class[] {String.class, Integer.class, Long.class, Float.class, Double.class, Byte.class, Short.class, Character.class, Boolean.class});
	@SuppressWarnings("rawtypes")
	private static final Collection scalarsArrays = Arrays.asList(new Class[] {String[].class, Integer[].class, Long[].class, Float[].class, Double[].class, Byte[].class, Short[].class, Character[].class, Boolean[].class});
	@SuppressWarnings("rawtypes")
	private static final Collection primitiveArrays = Arrays.asList(new Class[] {long[].class, int[].class, short[].class, char[].class, byte[].class, double[].class, float[].class, boolean[].class});

	@SuppressWarnings({ "rawtypes" })
	public static Map checkArgs(Map arguments, boolean validateValues) throws ApplicationException {
		if (arguments == null)
			return arguments;
		Map copy = validateValues ? new HashMap() : null;
		for (Iterator entries = arguments.entrySet().iterator(); entries.hasNext();) {
			Map.Entry entry = (Entry) entries.next();
			if (!(entry.getKey() instanceof String))
				throw new IllegalArgumentException("Invalid key type: " + entry.getKey() == null ? "<null>" : entry.getKey().getClass().getName());
			if ("".equals(entry.getKey())) //$NON-NLS-1$
				throw new IllegalArgumentException("Empty string is an invalid key");
			if (validateValues)
				validateValue(entry, copy);
		}
		return validateValues ? copy : arguments;
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	private static void validateValue(Map.Entry entry, Map copy) throws ApplicationException {
		Class clazz = entry.getValue().getClass();

		// Is it in the set of scalar types	
		if (scalars.contains(clazz)) {
			copy.put(entry.getKey(), entry.getValue());
			return;
		}

		// Is it an array of primitives or scalars
		if (scalarsArrays.contains(clazz) || primitiveArrays.contains(clazz)) {
			int arrayLength = Array.getLength(entry.getValue());
			Object copyOfArray = Array.newInstance(entry.getValue().getClass().getComponentType(), arrayLength);
			System.arraycopy(entry.getValue(), 0, copyOfArray, 0, arrayLength);
			copy.put(entry.getKey(), copyOfArray);
			return;
		}

		// Is it a Collection of scalars
		if (entry.getValue() instanceof Collection) {
			Collection valueCollection = (Collection) entry.getValue();
			for (Iterator it = valueCollection.iterator(); it.hasNext();) {
				Class containedClazz = it.next().getClass();
				if (!scalars.contains(containedClazz)) {
					throw new ApplicationException(ApplicationException.APPLICATION_INVALID_STARTUP_ARGUMENT, "The value for key \"" + entry.getKey() + "\" is a collection that contains an invalid value of type \"" + containedClazz.getName() + "\""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				}
			}
			copy.put(entry.getKey(), new ArrayList((Collection) entry.getValue()));
			return;
		}
		throw new ApplicationException(ApplicationException.APPLICATION_INVALID_STARTUP_ARGUMENT, "The value for key \"" + entry.getKey() + "\" is an invalid type \"" + clazz.getName() + "\""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	}

}
