/**
 * Copyright (c) 2012 - 2018 Data In Motion Consulting.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion Consulting - initial API and implementation
 */
package org.gecko.runtime.boot.internal;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.gecko.runtime.boot.GeckoBootConstants;
import org.osgi.annotation.bundle.Capability;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;

/**
 * Configurator that extracts system properties
 * @author Mark Hoffmann
 * @since 07.11.2017
 */
@Capability(
	namespace="gecko.addons",
	name="runtime",
	version="1.0.0"
	)
@Component(immediate=true, enabled=true)
public class BootConfigurator {
	
	private final Logger logger = Logger.getLogger("GeckoRuntime");
	private BundleContext bctx;
	private ServiceRegistration<URL> baseDirRegistration;
	private ServiceRegistration<URL> confDirRegistration;
	private ServiceRegistration<URL> dataDirRegistration;

	@Activate
	public void activate(ComponentContext ctx) {
		bctx = ctx.getBundleContext();
		String userHome = System.getProperty("user.home");
		baseDirRegistration = registerUrl(GeckoBootConstants.PROP_GECKO_BASE_DIR, userHome + "/.gecko");
		confDirRegistration = registerUrl(GeckoBootConstants.PROP_GECKO_CONFIG_DIR, userHome + "/.gecko/etc");
		dataDirRegistration = registerUrl(GeckoBootConstants.PROP_GECKO_DATA_DIR, userHome + "/.gecko/data");
	}
	
	@Deactivate
	public void deactivate() {
		if (baseDirRegistration != null) {
			baseDirRegistration.unregister();
		}
		if (confDirRegistration != null) {
			confDirRegistration.unregister();
		}
		if (dataDirRegistration != null) {
			dataDirRegistration.unregister();
		}
	}
	
	private ServiceRegistration<URL> registerUrl(String property, String defaultPath) {
		if (property == null) {
			throw new IllegalArgumentException("Cannot register a URL with an unknown property: '" + property + "'");
		}
		if (defaultPath == null) {
			defaultPath = "./";
		}
		String dirPath = bctx.getProperty(property);
		URL dirURL;
		if (dirPath == null) {
			dirPath = System.getenv(property);
		}
		if (dirPath == null) {
			dirPath = System.getProperty(property);
		}
		if (dirPath == null) {
			dirPath = defaultPath;
			logger.warning("No property '" + property + "' was provided, using path '" + dirPath + "'");
		}
		try {
			dirURL = new URL("file://" + dirPath);
			return registerUrlService(dirURL, property);
		} catch (MalformedURLException e) {
			logger.log(Level.SEVERE, "Cannot parse URL '" + dirPath + "'", e);
			return null;
		}
		
	}
	
	/**
	 * Registers the given URL with the given property as service
	 * @param url the url to register
	 * @param property the property that identifies the URL as service
	 * @return the {@link ServiceRegistration}
	 */
	private ServiceRegistration<URL> registerUrlService(URL url, String property) {
		if (url == null) {
			throw new IllegalArgumentException("Cannot register a null URL as service for property: '" + property + "'");
		}
		if (property == null) {
			throw new IllegalArgumentException("Cannot register a URL with an unknown property: '" + url.getPath() + "'");
		}
		logger.info("Registering Gecko property '" + property + "' for runtime URL '" + url.getPath() + "'");
		Dictionary<String, Object> properties = new Hashtable<String, Object>();
		properties.put(property, Boolean.TRUE);
		properties.put("url", url.toString());
		return bctx.registerService(URL.class, url, properties);
	}

}
