/**
 * Copyright (c) 2012 - 2018 Data In Motion Consulting.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion Consulting - initial API and implementation
 */
package org.gecko.runtime.config.internal;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.ArrayList;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.gecko.runtime.boot.GeckoBootConstants;
import org.gecko.runtime.config.GeckoConfigurationConstants;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;

/**
 * File watcher implementation for the configuration files
 * @author Mark Hoffmann
 */
public class ConfigurationFileWatcher implements Runnable {

	private final Logger logger = Logger.getLogger("GeckoRuntimeConfiguration");
	private final Map<String, ServiceRegistration<File>> configFileMap = new ConcurrentHashMap<String, ServiceRegistration<File>>();
	private BundleContext ctx;
	private URL configFileUrl;
	private WatchService confFileWatcher = null;
	private AtomicBoolean running = new AtomicBoolean(false);
	private FilenameFilter fileNameFilter = new ConfigurationFilenameFilter(new String[0]);

	public ConfigurationFileWatcher(BundleContext ctx, URL configFileUrl) {
		this.ctx = ctx;
		this.configFileUrl = configFileUrl;
	}

	/**
	 * Starts the file watcher
	 */
	public void start() {
		if (running.get()) {
			throw new IllegalStateException("Configuration watcher was already started");
		}
		try {
			running.set(true);
			Executors.newSingleThreadExecutor().submit(this);
		} catch (Exception e) {
			running.set(false);
			throw e;
		}
	}

	/**
	 * Stopps the watcher and disposes all resources
	 */
	public void stop() {
		disposeFolderWatcher();
		// clear all registrations
		configFileMap.keySet().forEach(this::unregisterFileService);
		configFileMap.clear();
		running.set(false);
	}

	/**
	 * Sets a list of file names that are valid to be watched
	 * @param filterList the list of files that can be watched
	 */
	public void setFilterList(String[] filterList) {
		if (filterList == null) {
			this.fileNameFilter = new ConfigurationFilenameFilter(new String[0]);
		} else {
			this.fileNameFilter = new ConfigurationFilenameFilter(filterList);
		}
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
		if (configFileUrl == null) {
			return;
		}
		while (running.get()) {
			File confFolder = registerFileWatcher();
			if (confFolder == null) {
				logger.warning(String.format("[%s] Registering the file watcher didn't succeed. File watcher stops running", configFileUrl));
				break;
			}
			readConfigDirectory();
			try {
				WatchKey key;
				while ((key = confFileWatcher.take()) != null) {
					if (!running.get()) {
						break;
					}
					if (!confFolder.exists()) {
						logger.log(Level.INFO, String.format("[%s] The config folder was obviously deleted in the meantime", GeckoBootConstants.PROP_GECKO_CONFIG_DIR));
						disposeFolderWatcher();
						break;
					}
					for (WatchEvent<?> event : key.pollEvents()) {
						Path p = (Path) event.context();
						File f = p.toFile();
						if (!fileNameFilter.accept(null, p.toFile().getName())) {
							continue;
						}
						if (event.kind().equals(StandardWatchEventKinds.ENTRY_CREATE) || 
								event.kind().equals(StandardWatchEventKinds.ENTRY_MODIFY)) {
							updateFileService(f);
						} else if (event.kind().equals(StandardWatchEventKinds.ENTRY_DELETE)) {
							unregisterFileService(f.getName());
						} else {
							logger.warning(String.format("[%] Detected an unsupported change of the file", event.context().toString()));
						}
					}
					key.reset();
				}
			} catch (InterruptedException e) {
				logger.log(Level.SEVERE, String.format("[%s] Watching the configuaration folder was interrupted", GeckoBootConstants.PROP_GECKO_CONFIG_DIR), e);
			}
		}
		stop();
	}

	/**
	 * Reads the configuration folder and registers all files in it as service
	 */
	private void readConfigDirectory() {
		if (configFileUrl == null) {
			logger.warning("No confioguration folder URL is available, to be read");
			return;
		}
		try {
			// Create a list of potential files, tho be deleted
			List<String> toBeDeleted = new ArrayList<String>(configFileMap.keySet());
			File confFolder = new File(configFileUrl.toURI());
			if (!confFolder.exists()) {
				logger.info(String.format("[%s] Detected an config folder that does not exist", confFolder.getAbsolutePath()));
				return;
			}
			if (confFolder.isDirectory()) {
				if (!confFolder.canRead()) {
					logger.info(String.format("[%s] Detected an config folder that is not readable. Stop further reading", confFolder.getName()));
					return;
				}
				for (File file : confFolder.listFiles(fileNameFilter)) {
					if (file.isDirectory()) {
						logger.info(String.format("[%s] Detected an folder", file.getName()));
						continue;
					}
					if (file.isHidden()) {
						logger.info(String.format("[%s] Detected an hidden file. It will be ignored", file.getName()));
						continue;
					}
					if (!file.canRead()) {
						logger.info(String.format("[%s] Detected an file with no read rights. It will be ignored", file.getName()));
						continue;
					}
					String currentFileName = file.getName();
					toBeDeleted.remove(currentFileName);
					updateFileService(file);
				}
				// Unregister all services that are left in the to be deleted list
				toBeDeleted.forEach(this::unregisterFileService);
			} else {
				logger.log(Level.SEVERE, String.format("[%s] The URL is expected to be a folder, but is obviously none", GeckoBootConstants.PROP_GECKO_CONFIG_DIR));
			}
		} catch (URISyntaxException e) {
			logger.log(Level.SEVERE, String.format("[%s] The URL cannot be converted into a file URI", GeckoBootConstants.PROP_GECKO_CONFIG_DIR), e);
		}
	}

	/**
	 * Registers the given {@link File} as service
	 * @param file the file to register
	 */
	private void updateFileService(File file) {
		if (file == null) {
			throw new IllegalArgumentException("Cannot register a null file as service");
		}
		ServiceRegistration<File> fileRegistration = null;
		String fileName = file.getName();
		Dictionary<String, Object> properties = new Hashtable<String, Object>();
		properties.put(GeckoConfigurationConstants.CONFIGURATION_FILE, Boolean.TRUE);
		properties.put(GeckoConfigurationConstants.CONFIGURATION_FILE_TIMESTAMP, file.lastModified());
		properties.put(GeckoConfigurationConstants.CONFIGURATION_FILE_NAME, fileName);
		if (configFileMap.containsKey(fileName)) {
			fileRegistration = configFileMap.get(fileName);
			if (fileRegistration != null) {
				fileRegistration.setProperties(properties);
				logger.info(String.format("[%s] Updated configuration file registration", fileName));
			}
		} else {
			fileRegistration =  ctx.registerService(File.class, file, properties);
			configFileMap.put(fileName, fileRegistration);
			logger.info(String.format("[%s] Registered configuration file", fileName));
		}
	}

	/**
	 * Unregisters a file service for the given name
	 * @param fileName the file name
	 */
	private void unregisterFileService(String fileName) {
		ServiceRegistration<File> fileRegistration = configFileMap.remove(fileName);
		if (fileRegistration != null) {
			logger.info(String.format("[%s] Un-registering configuration file", fileName));
			fileRegistration.unregister();
		}
	}

	/**
	 * Closes and cleans up the watcher 
	 */
	private void disposeFolderWatcher() {
		if (confFileWatcher != null) {
			try {
				confFileWatcher.close();
				confFileWatcher = null;
			} catch (IOException e) {
				logger.log(Level.SEVERE, String.format("[%s] The configuration folder watcher cannot be close", GeckoBootConstants.PROP_GECKO_CONFIG_DIR), e);
			}
		}
	}

	/**
	 * Registers the file watcher for the configuration path 
	 * @return the {@link File} instance
	 * @throws IOException
	 * @throws URISyntaxException
	 */
	private File registerFileWatcher() {
		try {
			if (confFileWatcher == null) {
				confFileWatcher = FileSystems.getDefault().newWatchService();
			}
			File confFolder = new File(configFileUrl.toURI());
			if (!confFolder.exists()) {
				confFolder.mkdirs();
				logger.log(Level.INFO, String.format("[%s] The config folder was created", GeckoBootConstants.PROP_GECKO_CONFIG_DIR));
			}
			if (!confFolder.isDirectory()) {
				logger.log(Level.SEVERE, String.format("[%s] The URL for the config path is expected to be a folder, but is obviously none", GeckoBootConstants.PROP_GECKO_CONFIG_DIR));
				return null;
			}
			Path confPath = confFolder.toPath();
			confPath.register(confFileWatcher, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE);
			return confFolder;
		} catch (URISyntaxException e) {
			logger.log(Level.SEVERE, String.format("[%s] The URL cannot be converted into a file URI for the watcher service", GeckoBootConstants.PROP_GECKO_CONFIG_DIR), e);
		} catch (IOException e) {
			logger.log(Level.SEVERE, String.format("[%s] The configuration folder cannot be registered for watching CREATE or REMOVE events", GeckoBootConstants.PROP_GECKO_CONFIG_DIR), e);
		}
		return null;
	}

}
