/**
 * Copyright (c) 2012 - 2018 Data In Motion Consulting.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion Consulting - initial API and implementation
 */
package org.gecko.runtime.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.IOException;

import org.gecko.runtime.boot.GeckoBootConstants;
import org.gecko.runtime.config.GeckoConfigurationConstants;
import org.gecko.util.test.common.service.ServiceChecker;
import org.gecko.util.test.common.test.AbstractOSGiTest;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;

/**
 * Integration test to test handling of the folder watcher in OSGi environments
 * @author Mark Hoffmann
 *
 */
@RunWith(MockitoJUnitRunner.class)
public class ExampleIntegrationTest extends AbstractOSGiTest {

	public ExampleIntegrationTest() {
		super(FrameworkUtil.getBundle(ExampleIntegrationTest.class).getBundleContext());
	}

	private File configFolder;

	@Before
	public void before() {
		String tmp = System.getProperty(GeckoBootConstants.PROP_GECKO_CONFIG_DIR);
		configFolder = new File(tmp);
		if (!configFolder.exists()) {
			configFolder.mkdirs();
		}
		assertTrue(configFolder.exists());
	}

	@After
	public void after() {
		assertTrue(configFolder.exists());
		if (configFolder.listFiles().length > 0) {
			for (File f : configFolder.listFiles()) {
				assertTrue(f.delete());
			}
		}
		assertTrue(configFolder.delete());
	}

	@Test
	public void testConfigFolderExists() throws InvalidSyntaxException {
		ServiceChecker<Object> configFileChecker = createdCheckerTrackedForCleanUp("(" + GeckoConfigurationConstants.CONFIGURATION_FILE + "=true)");
		configFileChecker.start();
		
		ServiceChecker<Object> configUrlChecker = createdCheckerTrackedForCleanUp("(" + GeckoBootConstants.PROP_GECKO_CONFIG_DIR + "=true)");
		configUrlChecker.start();
		assertEquals(1, configUrlChecker.getCurrentCreateCount(true));
		assertEquals(0, configFolder.listFiles().length);
		
		assertEquals(0, configFileChecker.getCurrentCreateCount(true));
		
		configUrlChecker.stop();
		configFileChecker.stop();
	}
	
	@Test
	public void testConfigFolderAddFile() throws InvalidSyntaxException, IOException {
		ServiceChecker<Object> configFileChecker = createdCheckerTrackedForCleanUp("(&(" + GeckoConfigurationConstants.CONFIGURATION_FILE + "=true)(" + GeckoConfigurationConstants.CONFIGURATION_FILE_NAME + "=test.conf))");
		configFileChecker.start();
		
		ServiceChecker<Object> configUrlChecker = createdCheckerTrackedForCleanUp("(" + GeckoBootConstants.PROP_GECKO_CONFIG_DIR + "=true)");
		configUrlChecker.start();
		assertEquals(1, configUrlChecker.getCurrentCreateCount(true));
		assertEquals(0, configFolder.listFiles().length);
		assertEquals(0, configFileChecker.getCurrentCreateCount(true));
		File test = new File(configFolder, "test.conf");
		assertTrue(test.createNewFile());
		assertEquals(1, configFileChecker.getCurrentCreateCount(true));
		
		configUrlChecker.stop();
		configFileChecker.stop();
		assertTrue(test.delete());
	}

}