/*******************************************************************************
 * Copyright (c) 2012, 2016 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.osgi.container;

import java.util.*;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.eclipse.osgi.util.ManifestElement;
import org.osgi.framework.*;
import org.osgi.framework.namespace.*;
import org.osgi.framework.wiring.*;
import org.osgi.resource.Capability;
import org.osgi.resource.Namespace;

/**
 * An implementation of {@link BundleRequirement}.  This requirement implements 
 * the matches method according to the OSGi specification which includes
 * implementing the mandatory directive for the osgi.wiring.* namespaces.
 * @since 3.10
 * @noextend This class is not intended to be subclassed by clients.
 */
public class ModuleRequirement implements BundleRequirement {

	public static final Pattern MANDATORY_ATTR = Pattern.compile("\\(([^(=<>]+)\\s*[=<>]\\s*[^)]+\\)"); //$NON-NLS-1$
	private final static Class<?> SINGLETON_MAP_CLASS = Collections.singletonMap(null, null).getClass();
	private final static Class<?> UNMODIFIABLE_MAP_CLASS = Collections.unmodifiableMap(Collections.emptyMap()).getClass();
	private final String namespace;
	private final Map<String, String> directives;
	private final Map<String, Object> attributes;

	ModuleRequirement(Map<String, String> directives) {
		this.namespace = IdentityNamespace.IDENTITY_NAMESPACE;
		this.directives = unmodifiableMap(directives);
		this.attributes = unmodifiableMap(new HashMap<String, Object>());
	}

	@Override
	public boolean matches(BundleCapability capability) {
		if (!namespace.equals(capability.getNamespace()))
			return false;
		String filterSpec = directives.get(Namespace.REQUIREMENT_FILTER_DIRECTIVE);
		Filter f = null;
		if (filterSpec != null) {
			try {
				f = FrameworkUtil.createFilter(filterSpec);
			} catch (InvalidSyntaxException e) {
				return false;
			}
		}
		boolean matchMandatory = PackageNamespace.PACKAGE_NAMESPACE.equals(namespace) || BundleNamespace.BUNDLE_NAMESPACE.equals(namespace) || HostNamespace.HOST_NAMESPACE.equals(namespace);
		return matches(f, capability, matchMandatory);
	}

	@Override
	public String getNamespace() {
		return namespace;
	}

	@Override
	public Map<String, String> getDirectives() {
		return directives;
	}

	@Override
	public Map<String, Object> getAttributes() {
		return attributes;
	}

	@Override
	public BundleRevision getRevision() {
		return null;
	}

	@Override
	public BundleRevision getResource() {
		return null;
	}

	@Override
	public String toString() {
		return namespace + toString(attributes, false) + toString(directives, true);
	}

	@SuppressWarnings("unchecked")
	static <K, V> Map<K, V> unmodifiableMap(Map<? extends K, ? extends V> map) {
		int size = map.size();
		if (size == 0) {
			return Collections.emptyMap();
		}
		if (size == 1) {
			if (map.getClass() != SINGLETON_MAP_CLASS) {
				Map.Entry<? extends K, ? extends V> entry = map.entrySet().iterator().next();
				map = Collections.<K, V> singletonMap(entry.getKey(), entry.getValue());
			}
		} else {
			if (map.getClass() != UNMODIFIABLE_MAP_CLASS) {
				map = Collections.unmodifiableMap(map);
			}
		}
		return (Map<K, V>) map;
	}

	static <V> String toString(Map<String, V> map, boolean directives) {
		return toString(map, directives, false);
	}

	static <V> String toString(Map<String, V> map, boolean directives, boolean stringsOnly) {
		if (map.size() == 0)
			return ""; //$NON-NLS-1$
		String assignment = directives ? ":=" : "="; //$NON-NLS-1$ //$NON-NLS-2$
		Set<Entry<String, V>> set = map.entrySet();
		StringBuilder sb = new StringBuilder();
		for (Entry<String, V> entry : set) {
			sb.append("; "); //$NON-NLS-1$
			String key = entry.getKey();
			Object value = entry.getValue();
			if (value instanceof List) {
				@SuppressWarnings("unchecked")
				List<Object> list = (List<Object>) value;
				if (list.size() == 0)
					continue;
				Object component = list.get(0);
				String className = component.getClass().getName();
				String type = className.substring(className.lastIndexOf('.') + 1);
				sb.append(key).append(':').append("List<").append(type).append(">").append(assignment).append('"'); //$NON-NLS-1$ //$NON-NLS-2$
				for (Object object : list)
					sb.append(object).append(',');
				sb.setLength(sb.length() - 1);
				sb.append('"');
			} else {
				String type = ""; //$NON-NLS-1$
				if (!(value instanceof String) && !stringsOnly) {
					String className = value.getClass().getName();
					type = ":" + className.substring(className.lastIndexOf('.') + 1); //$NON-NLS-1$
				}
				sb.append(key).append(type).append(assignment).append('"').append(value).append('"');
			}
		}
		return sb.toString();
	}

	public static boolean matches(Filter f, Capability candidate, boolean matchMandatory) {
		if (f != null && !f.matches(candidate.getAttributes())) {
			return false;
		}
		if (matchMandatory) {
			// check for mandatory directive
			String mandatory = candidate.getDirectives().get(AbstractWiringNamespace.CAPABILITY_MANDATORY_DIRECTIVE);
			if (mandatory == null) {
				return true;
			}
			if (f == null) {
				return false;
			}
			Matcher matcher = MANDATORY_ATTR.matcher(f.toString());
			String[] mandatoryAttrs = ManifestElement.getArrayFromList(mandatory, ","); //$NON-NLS-1$
			boolean allPresent = true;
			for (String mandatoryAttr : mandatoryAttrs) {
				matcher.reset();
				boolean found = false;
				while (matcher.find()) {
					int numGroups = matcher.groupCount();
					for (int i = 1; i <= numGroups; i++) {
						if (mandatoryAttr.equals(matcher.group(i))) {
							found = true;
						}
					}
				}
				allPresent &= found;
			}
			return allPresent;
		}
		return true;
	}

}
