/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.emf.osgi.json;

import static org.emfjson.jackson.databind.EMFContext.Attributes.RESOURCE;
import static org.emfjson.jackson.databind.EMFContext.Attributes.RESOURCE_SET;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.emfjson.jackson.annotations.EcoreIdentityInfo;
import org.emfjson.jackson.annotations.EcoreReferenceInfo;
import org.emfjson.jackson.annotations.EcoreTypeInfo;
import org.emfjson.jackson.databind.EMFContext;
import org.emfjson.jackson.handlers.URIHandler;
import org.emfjson.jackson.module.EMFModule;
import org.emfjson.jackson.module.EMFModule.Feature;
import org.emfjson.jackson.resource.JsonResource;
import org.gecko.emf.osgi.json.constants.EMFJs;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.cfg.ContextAttributes;

/**
 * 
 * @author jalbert
 * @since 27 Jun 2018
 */
public class ConfigurableJsonResource extends JsonResource {

	/**
	 * Creates a new instance.
	 */
	public ConfigurableJsonResource(URI uri) {
		super(uri, null);
	}
	
	public ConfigurableJsonResource(URI uri, ObjectMapper mapper) {
		super(uri, mapper);
	}
	
	public ObjectMapper createMapper(Map<?,?> options) {
		final ObjectMapper mapper = new ObjectMapper();
		
		String timeFormat = getOrDefault(options, EMFJs.OPTION_DATE_FORMAT, "yyyy-MM-dd'T'HH:mm:ss'Z'");
		final SimpleDateFormat dateFormat = new SimpleDateFormat(timeFormat, Locale.ENGLISH);
		dateFormat.setTimeZone(TimeZone.getDefault());
		
		boolean indentOutput = getOrDefault(options, EMFJs.OPTION_INDENT_OUTPUT, true);
		mapper.configure(SerializationFeature.INDENT_OUTPUT, indentOutput);
		mapper.setDateFormat(dateFormat);
		mapper.setTimeZone(TimeZone.getDefault());
		
		EMFModule module = new EMFModule();
		
		module.configure(Feature.OPTION_SERIALIZE_CONTAINMENT_AS_HREF, getOrDefault(options, EMFJs.OPTION_SERIALIZE_CONTAINMENT_AS_HREF, false));
		module.configure(Feature.OPTION_SERIALIZE_DEFAULT_VALUE, getOrDefault(options, EMFJs.OPTION_SERIALIZE_DEFAULT_VALUE, false));
		module.configure(Feature.OPTION_SERIALIZE_TYPE, getOrDefault(options, EMFJs.OPTION_SERIALIZE_TYPE, true));
		module.configure(Feature.OPTION_USE_ID, getOrDefault(options, EMFJs.OPTION_USE_ID, false));
		
		URIHandler uriHandler = (URIHandler) options.get(XMLResource.OPTION_URI_HANDLER);
		if(uriHandler != null) {
			module.setUriHandler(uriHandler);
		}
		String refField = getOrDefault(options, EMFJs.OPTION_REF_FIELD, null);
		String idField = getOrDefault(options, EMFJs.OPTION_ID_FIELD, null);
		String typeField = getOrDefault(options, EMFJs.OPTION_TYPE_FIELD, null);
		
		if(refField != null && uriHandler != null) {
			module.setReferenceInfo(new EcoreReferenceInfo(refField, uriHandler));
		} else if(refField != null && uriHandler == null) {
			module.setReferenceInfo(new EcoreReferenceInfo(refField));
		} else if (refField == null && uriHandler != null) {
			module.setReferenceInfo(new EcoreReferenceInfo(uriHandler));
		}
		
		if(typeField != null) {
			module.setTypeInfo(new EcoreTypeInfo(typeField));
		}
		
		if(idField != null) {
			module.setIdentityInfo(new EcoreIdentityInfo(idField));
		}
		
		mapper.registerModule(module);
		
		return mapper;
	}
	
	/**
	 * @param options
	 * @param optionDateFormat
	 * @param string
	 * @return
	 */
	@SuppressWarnings("unchecked")
	private <T> T getOrDefault(Map<?, ?> options, String key, T defaultvalue) {
		Object value = options.get(key);
		if(value == null) {
			return defaultvalue;
		}
		return (T) value;
	}

	@Override
	protected void doLoad(InputStream inputStream, Map<?, ?> options) throws IOException {
		if (options == null) {
			options = Collections.<String, Object>emptyMap();
		}

		if (inputStream instanceof URIConverter.Loadable) {

			((URIConverter.Loadable) inputStream).loadResource(this);

		} else {

			ContextAttributes attributes = EMFContext
					.from(options)
					.withPerCallAttribute(RESOURCE_SET, getResourceSet())
					.withPerCallAttribute(RESOURCE, this);

			
			createMapper(options).reader()
					.with(attributes)
					.forType(Resource.class)
					.withValueToUpdate(this)
					.readValue(inputStream);

		}
	}

	@Override
	protected void doSave(OutputStream outputStream, Map<?, ?> options) throws IOException {
		if (options == null) {
			options = Collections.<String, Object>emptyMap();
		}

		if (outputStream instanceof URIConverter.Saveable) {

			((URIConverter.Saveable) outputStream).saveResource(this);

		} else {

			createMapper(options).writer()
					.with(EMFContext.from(options))
					.writeValue(outputStream, this);

		}
	}
	
}
