/**
 * Copyright (c) 2012 - 2017 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.emf.osgi.json;

import java.text.SimpleDateFormat;
import java.util.Locale;
import java.util.TimeZone;

import org.eclipse.emf.ecore.resource.Resource.Factory.Registry;
import org.emfjson.jackson.module.EMFModule;
import org.emfjson.jackson.resource.JsonResourceFactory;
import org.gecko.emf.osgi.ResourceFactoryConfigurator;
import org.gecko.emf.osgi.annotation.EMFResourceFactoryConfigurator;
import org.gecko.emf.osgi.annotation.provide.ProvideEMFResourceConfigurator;
import org.osgi.service.component.annotations.Component;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;

/**
 * Implementation of the {@link IResourceSetConfigurator} to provide support for {@link JsResourceImpl}.
 * 
 * It provides the {@link JsResourceFactoryImpl} for the following identifiers:
 * <ul>
 * 	<li>Extension: json
 * 	<li>contentType: application/json
 * 	<li>contentType: application/x-json
 * 	<li>Protocol: json
 * </ul>
 * 
 * @author Juergen Albert
 * @since 27.06.2014
 */
@Component(name="EMFJsonConfigurator", immediate=true, service=ResourceFactoryConfigurator.class)
@ProvideEMFResourceConfigurator(name="EMFJson", 
	contentType = {
			"application/yml",
			"application/yaml",
			"application/json",
			"application/x-json",
			"application/emf-json"
	},
	fileExtension = {
			"json",
			"yml",
			"yaml"
	},
	version = "1.0.1"
)
@EMFResourceFactoryConfigurator(
		emf_resource_configurator_name ="EMFJson",
		emf_resource_configurator_fileExtension = {
				"json",
				"yml",
				"yaml"
		},
		emf_resource_configurator_contentType = {
				"application/yml",
				"application/yaml",
				"application/json",
				"application/x-json",
				"application/emf-json"
			}
		)
public class EMFJSONResourceFactoryConfigurator implements ResourceFactoryConfigurator{

	/* 
	 * (non-Javadoc)
	 * @see de.dim.emf.osgi.ResourceFactoryConfigurator#configureResourceFactory(org.eclipse.emf.ecore.resource.Resource.Factory.Registry)
	 */
	@Override
	public void configureResourceFactory(Registry registry) {

		registry.getExtensionToFactoryMap().put("yml", createYamlFactory());
		registry.getExtensionToFactoryMap().put("yaml", createYamlFactory());
		registry.getContentTypeToFactoryMap().put("application/yml", createYamlFactory());
		registry.getContentTypeToFactoryMap().put("application/yaml", createYamlFactory());
		
		registry.getExtensionToFactoryMap().put("json", new JsonResourceFactory());
		registry.getProtocolToFactoryMap().put("json", new JsonResourceFactory());
		registry.getContentTypeToFactoryMap().put("application/json", new JsonResourceFactory());
		registry.getContentTypeToFactoryMap().put("application/x-json", new JsonResourceFactory());
		registry.getContentTypeToFactoryMap().put("application/emf-json", new JsonResourceFactory());
	}
	
	private JsonResourceFactory createYamlFactory() {
		ObjectMapper mapper = new ObjectMapper(new YAMLFactory());
		final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss", Locale.ENGLISH);
		dateFormat.setTimeZone(TimeZone.getDefault());

		mapper.configure(SerializationFeature.INDENT_OUTPUT, true);
		mapper.setDateFormat(dateFormat);
		mapper.setTimeZone(TimeZone.getDefault());
		mapper.registerModule(new EMFModule());
		
		return new JsonResourceFactory(mapper);
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.emf.osgi.ResourceFactoryConfigurator#unconfigureResourceFactory(org.eclipse.emf.ecore.resource.Resource.Factory.Registry)
	 */
	@Override
	public void unconfigureResourceFactory(Registry registry) {
		registry.getExtensionToFactoryMap().remove("yml");
		registry.getExtensionToFactoryMap().remove("yaml");
		registry.getContentTypeToFactoryMap().remove("application/yml");
		registry.getContentTypeToFactoryMap().remove("application/yaml");
		
		registry.getExtensionToFactoryMap().remove("json");
		registry.getContentTypeToFactoryMap().remove("application/json");
		registry.getContentTypeToFactoryMap().remove("application/x-json");
		registry.getContentTypeToFactoryMap().remove("application/emf-json");
		registry.getProtocolToFactoryMap().remove("json");
	}

}
