/**
 * Copyright (c) 2014 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.emf.util.registry;

import java.io.IOException;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.gecko.emf.util.registry.exceptions.RegistryException;
import org.gecko.util.uriprovider.LocationUriProvider;

/**
 * Implementation for an abstract registry that handles initializing and unregistering on close.
 * The registry of type R can register objects of type T. The registry is persisted 
 * @author Mark Hoffmann
 * @since 05.08.2014
 */
public abstract class DefaultRegistry <T extends EObject, R extends EObject> {
	
	private final List<RegistryIntializer<?>> initializer = new LinkedList<RegistryIntializer<?>>();
	private ResourceSet resourceSet = null;
	private LocationUriProvider uriProvider;
	private R registry = null;
	
	/**
	 * Adds the given object to the given registry and returns <code>true</code>, on success otherwise <code>false</code>
	 * @param registry the registry to add the object to, cannot be <code>null</code>
	 * @param object the object to register, cannot be <code>null</code>
	 * @return <code>true</code>, on successful registration, otherwise <code>false</code>
	 */
	abstract public boolean addToRegistry(R registry, T object);

	/**
	 * Removes the given object from the given registry and returns <code>true</code>, on success otherwise <code>false</code>
	 * @param registry the registry to remove the object to, cannot be <code>null</code>
	 * @param object the object to remove, cannot be <code>null</code>
	 * @return <code>true</code>, on successful un-registration, otherwise <code>false</code>
	 */
	abstract public boolean removeFromRegistry(R registry, T object);
	
	/**
	 * Returns all registered objects or an empty list
	 * @return all registered objects or an empty list
	 */
	public List<T> getAllRegisteredObjects() {
		List<T> result = getAllObject(getRegistry());
		return result == null ? Collections.<T>emptyList() : Collections.unmodifiableList(result);
	}

	/**
	 * Registers the object and returns <code>true</code> on success
	 * otherwise <code>false</code>, if e.g. the object already exists or is <code>null</code>
	 * @param object the object to register
	 * @return <code>true</code> on success otherwise <code>false</code>
	 */
	public boolean registerObject(T object) {
		if (object == null) {
			return false;
		}
		boolean result = addToRegistry(getRegistry(), object);
		saveRegistry();
		return result;
	}

	/**
	 * Unregisters the object and returns <code>true</code> on success
	 * otherwise <code>false</code>.
	 * @param object the object to un-register
	 * @return <code>true</code> on success otherwise <code>false</code>
	 */
	public boolean unregisterObject(T object) {
		if (object == null) {
			return false;
		}
		boolean result = removeFromRegistry(getRegistry(), object);
		saveRegistry();
		return result;
	}
	
	
	
	/**
	 * Adds a new {@link RegistryIntializer} to the registry
	 * @param initializer the initializer to add
	 */
	public void addInitializer(RegistryIntializer<T> initializer) {
		if (initializer == null) {
			return;
		}
		this.initializer.add(initializer);
		List<T> objects = initializer.initializeRegistry();
		if (objects == null) {
			return;
		}
		for (T object : objects) {
			addToRegistry(getRegistry(), object);
		}
		if (objects.size() > 0) {
			saveRegistry();
		}
	}

	/**
	 * Removes a new {@link RegistryIntializer} from the registry
	 * @param initializer the initializer to remove
	 */
	public void removeInitializer(RegistryIntializer<T> initializer) {
		if (initializer == null) {
			return;
		}
		this.initializer.remove(initializer);
		List<T> objects = initializer.disposeRegistry();
		if (objects == null) {
			return;
		}
		
		for (T object : objects) {
			removeFromRegistry(getRegistry(), object);
		}
		if (objects.size() > 0) {
			saveRegistry();
		}
	}

	/**
	 * Returns the uri provider instance
	 * @return the uri provider instance
	 */
	public LocationUriProvider getLocationUriProvider() {
		return uriProvider;
	}

	/**
	 * Sets the {@link LocationUriProvider} instance
	 * @param uriProvider the instance to set
	 */
	public void setLocationUriProvider(LocationUriProvider uriProvider) {
		this.uriProvider = uriProvider;
	}

	/**
	 * Disposes the registry
	 */
	public void dispose() {
		saveRegistry();
		registry.eResource().unload();
		resourceSet.getResources().remove(registry.eResource());
	}
	
	/**
	 * Returns all objects of the given registry
	 * @param registry the registry
	 * @return a list of objects, or <code>null</code>
	 */
	abstract protected List<T> getAllObject(R registry);

	/**
	 * Initializes the resource set if necessary
	 * @param resourceSet the resource set to initialize
	 */
	abstract protected void initializeResourceSet(ResourceSet resourceSet);
	
	/**
	 * Creates an emtpy instance of the registry
	 * @return an emtpy instance of the registry
	 */
	abstract protected R createEmptyRegistryInstance();

	/**
	 * Returns the {@link ResourceSet}
	 * @return the {@link ResourceSet}
	 */
	protected ResourceSet getResourceSet() {
		if (resourceSet == null) {
			resourceSet = new ResourceSetImpl();
		}
		return resourceSet;
	}
	
	/**
	 * Returns the registry object and initializes it lazy
	 * @return the registry object
	 */
	protected R getRegistry() {
		if (registry == null) {
			initRegistry();
		}
		return registry;
	}
	
	/**
	 * Initializes the registry
	 */
	@SuppressWarnings("unchecked")
	protected void initRegistry() {
		Resource registryResource = null;
		initializeResourceSet(getResourceSet());
		try {
			if (getLocationUriProvider() == null) {
				throw new RegistryException("Cannot initialize registry without location uri provider");
			}
			String registryUri = getLocationUriProvider().getLocationUri();
			if (registryUri == null) {
				throw new RegistryException("Cannot initialize registry with a null registry path");
			}
			URI uri = URI.createURI(registryUri);
			registryResource = resourceSet.createResource(uri);
			if (registryResource == null) {
				throw new RegistryException("Error creating resource for uri (Maybe we are not in an OSGi environment an the factories are not registered) uri: " + uri);
			}
			registryResource.load(null);
			if (registryResource.getContents().size() > 0) {
				registry = (R) registryResource.getContents().get(0);
			} else {
				throw new RegistryException("There is no content in the registry resource");
			}
		} catch (IOException e) {
			registry = createEmptyRegistryInstance();
			registryResource.getContents().add(registry);
			saveRegistry();
		} catch (Exception e) {
			throw new RegistryException("Error initializing registry", e);
		}
		
	}
	
	/**
	 * Saves the registry
	 */
	protected void saveRegistry() {
		if (registry.eResource() != null) {
			try {
				registry.eResource().save(null);
			} catch (IOException e) {
				throw new RegistryException("Error saving the registry", e);
			}
		} else {
			throw new RegistryException("Registry object doesn't have any resource to save");
		}
	}

}
