/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.util.pac4j.clients;

import java.util.Map;
import java.util.function.Consumer;

import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.pac4j.core.client.Client;
import org.pac4j.oidc.config.KeycloakOidcConfiguration;

import com.nimbusds.jose.JWSAlgorithm;

/**
 * The {@link KeycloackOidClient} is an OID Client, performning a OIDC login action if necessary.
 * @author Juergen Albert
 * @since 10 Sep 2018
 */
@Component(service = Client.class, name = "KeycloackOidcClient", configurationPolicy = ConfigurationPolicy.REQUIRE)
public class KeycloackOidcClient extends org.pac4j.oidc.client.KeycloakOidcClient{

	@Activate
	public void activate(Map<String, Object> properties) throws ConfigurationException {
		
		KeycloakOidcConfiguration config = new KeycloakOidcConfiguration();
		setParameterIfAvailable(config::setBaseUri, properties, "oid.baseUri", true);
		setParameterIfAvailable(config::setRealm, properties, "oid.realm", true);
		setParameterIfAvailable(config::setClientId, properties, "oid.clientId", true);
		setParameterIfAvailable(config::setSecret, properties, "oid.secret", true);
		setParameterIfAvailable(this::setName, properties, "client.id", true);
		setParameterIfAvailable(this::setCallbackUrl, properties, "client.callbackUrl", true);
		if(properties.containsKey("oid.jws.algorithm")) {
			String alg = properties.get("oid.jws.algorithm").toString();
			JWSAlgorithm jwsAlgorithm = JWSAlgorithm.parse(alg);
			config.setPreferredJwsAlgorithm(jwsAlgorithm);
		}
		config.setExpireSessionWithToken(true);
		setConfiguration(config);
	}
	
	
	
	/**
	 * @param object
	 * @param properties
	 * @param string
	 * @param b
	 * @throws ConfigurationException 
	 */
	private void setParameterIfAvailable(Consumer<String> function, Map<String, Object> properties, String key, boolean mandatory) throws ConfigurationException {
		Object value = properties.get(key);
		if(value == null) {
			if(mandatory) {
				throw new ConfigurationException(key, "Mandatory");
			} else {
				return;
			}
		}
		function.accept(value.toString());
	}
}
