/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.util.rest.pac4j.feature.clients;

import java.text.ParseException;
import java.util.Map;
import java.util.function.Consumer;

import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.pac4j.core.client.Client;
import org.pac4j.core.client.DirectClient;
import org.pac4j.core.context.WebContext;
import org.pac4j.core.credentials.extractor.CredentialsExtractor;
import org.pac4j.oidc.config.KeycloakOidcConfiguration;
import org.pac4j.oidc.credentials.OidcCredentials;
import org.pac4j.oidc.profile.OidcProfile;
import org.pac4j.oidc.profile.creator.OidcProfileCreator;

import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jwt.JWTParser;

/**
 * The {@link BearerTokenClient} is an OID Client, that extracts the Bearer Token from the Authorization header. it expects an JWT ID Token here.
 * The Token itself will be validated internally without asking the OID Authority again. 
 * @author Juergen Albert
 * @since 10 Sep 2018
 */
@Component(service = Client.class, name = "BearerTokenClient", configurationPolicy = ConfigurationPolicy.REQUIRE)
public class BearerTokenClient extends DirectClient<OidcCredentials, OidcProfile>{

	@Activate
	public void activate(Map<String, Object> properties) throws ConfigurationException {
		
		KeycloakOidcConfiguration bearerconfig = new KeycloakOidcConfiguration();
		setParameterIfAvailable(bearerconfig::setBaseUri, properties, "oid.baseUri", true);
		setParameterIfAvailable(bearerconfig::setRealm, properties, "oid.realm", true);
		setParameterIfAvailable(bearerconfig::setClientId, properties, "oid.clientId", true);
		setParameterIfAvailable(bearerconfig::setSecret, properties, "oid.secret", true);
		setParameterIfAvailable(this::setName, properties, "client.id", true);
		if(properties.containsKey("oid.jws.algorithm")) {
			String alg = properties.get("oid.jws.algorithm").toString();
			JWSAlgorithm jwsAlgorithm = JWSAlgorithm.parse(alg);
			bearerconfig.setPreferredJwsAlgorithm(jwsAlgorithm);
		}
		//set the mandatory stuff
		setProfileCreator(new OidcProfileCreator<OidcProfile>(bearerconfig));
		setAuthenticator(new OidcBearerAuthenticator(bearerconfig));		
		setCredentialsExtractor(new CredentialsExtractor<OidcCredentials>() {
			
			@Override
			public OidcCredentials extract(WebContext context) {
				
				String authorization = context.getRequestHeader("Authorization");
				OidcCredentials oidcCredentials = new OidcCredentials();
				
				if(authorization != null) {
					authorization = authorization.substring("Bearer ".length());
					try {
						oidcCredentials.setIdToken(JWTParser.parse(authorization));
					} catch (ParseException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}
				}
				return oidcCredentials;
			}
		});
	}
	
	
	
	/**
	 * @param object
	 * @param properties
	 * @param string
	 * @param b
	 * @throws ConfigurationException 
	 */
	private void setParameterIfAvailable(Consumer<String> function, Map<String, Object> properties, String key, boolean mandatory) throws ConfigurationException {
		Object value = properties.get(key);
		if(value == null) {
			if(mandatory) {
				throw new ConfigurationException(key, "Mandatory");
			} else {
				return;
			}
		}
		function.accept(value.toString());
	}



	/* 
	 * (non-Javadoc)
	 * @see org.pac4j.core.client.DirectClient#clientInit()
	 */
	@Override
	protected void clientInit() {
		
	}
	
	

}
