/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.util.test.common.test;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import java.io.IOException;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.gecko.util.test.common.service.ServiceChecker;
import org.junit.After;
import org.junit.Assert;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Filter;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.util.tracker.ServiceTracker;

/**
 * A basic Test for OSGi. It provides convinient access to Services and manages cleanup opperations, when services are registered via the internal methods. 
 * @author Juergen Albert
 * @since 15 Jun 2018
 */
/**
 * 
 * @author jalbert
 * @since 18 Jun 2018
 */
public abstract class AbstractOSGiTest {

	private BundleContext bundleContext;

	Map<Object,ServiceReference<?>> referencesToCleanup = new HashMap<>();
	Map<Object, ServiceRegistration<?>> registrationsToCleanup = new HashMap<>();
	Map<Object, ServiceTracker<?, ?>> serviceTrackerToCleanUp = new HashMap<>();
	
	List<Configuration> configsToClean = new LinkedList<>();
	List<ServiceChecker<?>> waitForRemoves = new LinkedList<>();

	private ConfigurationAdmin configAdmin;
	
	/**
	 * Creates a new instance.
	 */
	public AbstractOSGiTest(BundleContext bundleContext) {
		this.bundleContext = bundleContext;
		configAdmin = getService(ConfigurationAdmin.class);
	}

	/**
	 * Cleanly ungets the given service, if it was retrieved via the getService method.
	 * @param service the service to unget
	 */
	protected void ungetService(Object service) {
		ServiceReference<?> toUnget = referencesToCleanup.remove(service);
		if(toUnget != null) {
			getBundleContext().ungetService(toUnget);
		} else {
			ServiceTracker<?,?> serviceTracker = serviceTrackerToCleanUp.remove(service);
			if(serviceTracker != null) {
				serviceTracker.close();
			}
		}
	}
	
	/**
	 * This method will retrieve a Service and does a {@link Assert#assertNotNull(Object)} check. When the test ends, the service will be unget as well. 
	 * @param clazz The Service Class desired
	 * @return the service will never be null 
	 */
	protected <T> T getService (Class<T> clazz) {
		ServiceReference<T> serviceReference = bundleContext.getServiceReference(clazz);
		assertNotNull(serviceReference);
		T service = bundleContext.getService(serviceReference);
		referencesToCleanup.put(service, serviceReference);
		return service;
	}

	/**
	 * This method will retrieve a Service and does a {@link Assert#assertNotNull(Object)} check. When the test ends, the service will be unget as well. 
	 * @param filter the {@link Filter} to use 
	 * @return the service, will never be null
	 * @throws InterruptedException 
	 */
	protected <T> T getService (Filter filter, long timeout) throws InterruptedException {
		ServiceTracker<T, T> tracker = new ServiceTracker<>(getBundleContext(), filter, null);
		tracker.open();
		T service = tracker.waitForService(timeout);
		serviceTrackerToCleanUp.put(service, tracker);
		assertNotNull(service);
		return service;
	}

	/**
	 * This method will try to a Service and does a {@link Assert#assertNull(Object)} check. 
	 * @param filter the {@link Filter} to use 
	 * @throws InterruptedException 
	 */
	protected void getServiceAssertNull(Filter filter) throws InterruptedException {
		ServiceTracker<?, ?> tracker = new ServiceTracker<>(getBundleContext(), filter, null);
		tracker.open();
		Object service = tracker.waitForService(50);
		serviceTrackerToCleanUp.put(new Object(), tracker);
		assertNull(service);
	}
	
	/**
	 * A configuration is created and update will be called. The config is furthermore stored for later cleanup.
	 * @param factoryPid the factoryPid of the service
	 * @param location the location of the configuration
	 * @param props the properties for the service
	 * @return the config you desire
	 * @throws IOException
	 */
	protected Configuration createConfigForCleanup(String factoryPid, String location, Dictionary<String, ?> props) throws IOException {
		Configuration config = configAdmin.createFactoryConfiguration(factoryPid, "?");
		configsToClean.add(config);
		config.update(props);
		return config;
	}
	
	/**
	 * The {@link Configuration} to delete
	 * @param config the {@link Configuration} to remove
	 * @throws IOException
	 */
	protected void deleteConfigurationAndRemoveFromCleanup(Configuration config) throws IOException {
		configsToClean.remove(config);
		config.delete();
	}

	/**
	 * The {@link Configuration} to delete. The method will block, until it receives the notification, that the service is removed. 
	 * The method will throw a {@link RuntimeException} after 5 seconds if nothing happens.
	 * @param config the {@link Configuration} to remove
	 * @throws IOException
	 * @throws InterruptedException 
	 */
	protected void deleteConfigurationAndRemoveFromCleanupBlocking(Configuration config) throws IOException, InterruptedException {
		configsToClean.remove(config);
		Filter filter = createFilter(config.getProperties()); 
		ServiceChecker<?> checker = new ServiceChecker<>(filter, getBundleContext());
		checker.setDeleteCount(1);
		checker.setDeleteTimeout(5);
		checker.start();
		config.delete();
		if(!checker.waitRemove()) {
			throw new RuntimeException("Service Remove never appeared");
		};
		checker.stop();
		return;
	}
	
	
	/**
	 * Creates a filter String looking for all properties in the {@link Dictionary} given.
	 * @param properties the {@link Dictionary} with the service properties.
	 * @return the {@link Filter} Object
	 */
	protected Filter createFilter(Dictionary<String, Object> properties){
		StringBuilder sb = new StringBuilder("(");
		Enumeration<String> keys = properties.keys();
		while(keys.hasMoreElements()) {
			String key = keys.nextElement();
			Object value = properties.get(key);
			sb.append(String.format("(%s=%s)", key, value.toString()));
		}
		sb.append(")");
		try {
			return FrameworkUtil.createFilter(sb.toString());
		} catch (InvalidSyntaxException e) {
			throw new IllegalArgumentException("Could not create filter " + e.getFilter(), e);
		}
	}

	/**
	 * Creates a ServiceChecker for a specific Service, where at least one Remove is expected. 
	 * At the End of the Test, it will block until the service is definitly gone.
	 * @param filter the OSGi filter String to create the checker with
	 * @return the {@link ServiceChecker}
	 * @throws InvalidSyntaxException if the filter syntax is wrong
	 */
	protected <T extends Object> ServiceChecker<T> createdCheckerTrackedForCleanUp(Class<T> serviceClass) {
		ServiceChecker<T> checker = new ServiceChecker<>(serviceClass, getBundleContext());
		
		checker.setCreateCount(1);
		checker.setDeleteCount(1);
		checker.setCreateTimeout(5);
		checker.setDeleteTimeout(5);
		waitForRemoves.add(checker);
		return (ServiceChecker<T>) checker;
	}

	/**
	 * Creates a ServiceChecker for a specific Service, where at least one Remove is expected. 
	 * At the End of the Test, it will block until the service is definitly gone.
	 * @param filter the OSGi filter String to create the checker with
	 * @return the {@link ServiceChecker}
	 * @throws InvalidSyntaxException if the filter syntax is wrong
	 */
	protected <T extends Object> ServiceChecker<T>  createdCheckerTrackedForCleanUp(String filter) throws InvalidSyntaxException {
		ServiceChecker<T> checker = new ServiceChecker<>(filter, getBundleContext());
		
		checker.setCreateCount(1);
		checker.setDeleteCount(1);
		checker.setCreateTimeout(5);
		checker.setDeleteTimeout(5);
		waitForRemoves.add(checker);
		return (ServiceChecker<T>) checker;
	}
	
	/**
	 * Cleanup Method
	 * @throws Exception
	 */
	@After
	public void after() throws Exception {
		System.out.println();
		System.out.println("============================================================");
		System.out.println("=================== Cleaning up ============================");
		System.out.println("============================================================");
		System.out.println();
		try {
			configsToClean.forEach(c -> {
				try {
					c.delete();
				} catch (IOException e) {
					assertNull(e);
				}
			});
		} catch (Exception e) {
			assertNull(e);
		}
		
		configAdmin = null;
		referencesToCleanup.forEach((o, s) -> bundleContext.ungetService(s));
		referencesToCleanup.clear();
		registrationsToCleanup.forEach((o, s) -> s.unregister());
		registrationsToCleanup.clear();

		serviceTrackerToCleanUp.forEach((o, s) -> s.close());
		serviceTrackerToCleanUp.clear();
		
		waitForRemoves.forEach(sc -> {
			try {
				sc.waitRemove();
			} catch (InterruptedException e) {
				assertNull(e);
			} finally {
				sc.stop();
			}
		});
		
		
		System.out.println();
		System.out.println("============================================================");
		System.out.println("=================== Cleaning finished ======================");
		System.out.println("============================================================");
		System.out.println();
	}
	
	/**
	 * Returns the configAdmin.
	 * @return the configAdmin
	 */
	public ConfigurationAdmin getConfigAdmin() {
		return configAdmin;
	}
	
	/**
	 * Returns the bundleContext.
	 * @return the bundleContext
	 */
	public BundleContext getBundleContext() {
		return bundleContext;
	}

}
