package org.gecko.emf.mongo.handlers;

import java.util.Dictionary;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.gecko.emf.mongo.InputStreamFactory;
import org.gecko.emf.mongo.OutputStreamFactory;
import org.gecko.emf.osgi.EMFNamespaces;
import org.gecko.emf.osgi.ResourceSetConfigurator;
import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;

/**
 * This implementation of the ResourceSetConfigurator service will attach
 * all currently bound URI handlers to the ResourceSet. This service is
 * intended to be used with the IResourceSetFactory service.
 * 
 * @author bhunt
 * 
 */
@Component(name="MongoResourceSetConfiguratorComponent", immediate=true)
public class MongoResourceSetConfiguratorComponent {

	public static final String PROP_MONGO_ALIAS = "database.alias";
	private final Map<String, MongoDatabaseProvider> mongoDatabaseProviders = new ConcurrentHashMap<String, MongoDatabaseProvider>();
	private ServiceRegistration<ResourceSetConfigurator> configuratorRegistration;
	private MongoURIHandlerProvider uriHandlerProvider = new MongoURIHandlerProvider();
	private BundleContext ctx;
	private List<String> aliases = new LinkedList<String>();

	/**
	 * Called on component activation
	 * @param context the component context
	 */
	@Activate
	public void activate(ComponentContext context) {
		ctx = context.getBundleContext();
		Dictionary<String, Object> properties = getDictionary();
		configuratorRegistration = ctx.registerService(ResourceSetConfigurator.class, new MongoResourceSetConfigurator(uriHandlerProvider), properties);
		System.out.println("activate Mongo resource set configurator");
	}

	/**
	 * Called on component deactivation
	 */
	@Deactivate
	public void deactivate() {
		configuratorRegistration.unregister();
		configuratorRegistration = null;
	}

	/**
	 * Adds a {@link MongoDatabaseProvider} to the provider map.  
	 * @param mongoDatabaseProvider the provider to be added
	 */
	@Reference(name="MongoDatabaseProvider", policy=ReferencePolicy.DYNAMIC, cardinality=ReferenceCardinality.AT_LEAST_ONE, unbind="removeMongoDatabaseProvider")
	public void addMongoDatabaseProvider(MongoDatabaseProvider mongoDatabaseProvider, Map<String, Object> properties) {
		String uri = mongoDatabaseProvider.getURI();
		mongoDatabaseProviders.put(uri, mongoDatabaseProvider);
		System.out.println("add database provider " + uri);
		uriHandlerProvider.addMongoDatabaseProvider(mongoDatabaseProvider);
		updateProperties(MongoDatabaseProvider.PROP_ALIAS, properties, true);
	}

	/**
	 * Removes a {@link MongoDatabaseProvider} from the map 
	 * @param mongoDatabaseProvider the provider to be removed
	 */
	public void removeMongoDatabaseProvider(MongoDatabaseProvider mongoDatabaseProvider, Map<String, Object> properties) {
		String uri = mongoDatabaseProvider.getURI();
		mongoDatabaseProviders.remove(uri);
		uriHandlerProvider.removeMongoDatabaseProvider(mongoDatabaseProvider);
		System.out.println("remove database provider " + uri);
		updateProperties(MongoDatabaseProvider.PROP_ALIAS, properties, false);
	}

	/**
	 * Sets an {@link InputStreamFactory} to handle input streams
	 * @param inputStreamFactory the factory to set
	 */
	@Reference(name="InputStreamFactory", cardinality=ReferenceCardinality.MANDATORY, policy=ReferencePolicy.STATIC)
	public void setInputStreamFactory(InputStreamFactory inputStreamFactory) {
		uriHandlerProvider.setInputStreamFactory(inputStreamFactory);
	}

	/**
	 * Sets an {@link OutputStreamFactory} to handle output streams
	 * @param outputStreamFactory the factory to set
	 */
	@Reference(name="OutputStreamFactory", cardinality=ReferenceCardinality.MANDATORY, policy=ReferencePolicy.STATIC)
	public void setOutputStreamFactory(OutputStreamFactory outputStreamFactory) {
		uriHandlerProvider.setOutputStreamFactory(outputStreamFactory);
	}

	/**
	 * Updates the properties of the service, depending on changes on injected services
	 * @param type the type of the property to publish 
	 * @param serviceProperties the service properties from the injected service
	 * @param add <code>true</code>, if the service was add, <code>false</code> in case of an remove
	 */
	private void updateProperties(String type, Map<String, Object> serviceProperties, boolean add) {
		Object name = serviceProperties.get(type);
		if (name != null && name instanceof String) {
			switch (type) {
			case MongoDatabaseProvider.PROP_ALIAS:
				if (add) {
					aliases.add(name.toString());
				} else {
					aliases.remove(name.toString());
				}
				break;
			default:
				break;
			}
			updateRegistrationProperties();
		}
	}

	/**
	 * Updates the service registration properties
	 */
	private void updateRegistrationProperties() {
		if (configuratorRegistration != null) {
			configuratorRegistration.setProperties(getDictionary());
		}
	}

	/**
	 * Creates a dictionary for the stored properties
	 * @return a dictionary for the stored properties
	 */
	private Dictionary<String, Object> getDictionary() {
		Dictionary<String, Object> properties = new Hashtable<>();
		String[] aliasNames = aliases.toArray(new String[0]);
		if (aliasNames.length > 0) {
			properties.put(PROP_MONGO_ALIAS, aliasNames);
		}
		properties.put(EMFNamespaces.EMF_CONFIGURATOR_NAME, "mongo");
		return properties;
	}
	
}
