/*******************************************************************************
 * Copyright (c) 2012 Bryan Hunt.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Bryan Hunt - initial API and implementation
 *******************************************************************************/

package org.gecko.emf.mongo.streams;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.bson.Document;
import org.eclipse.emf.common.util.URI;
import org.gecko.emf.mongo.ConverterService;
import org.gecko.emf.mongo.InputContentHandler;
import org.gecko.emf.mongo.InputStreamFactory;
import org.gecko.emf.mongo.OutputStreamFactory;
import org.gecko.emf.mongo.QueryEngine;
import org.gecko.mongo.osgi.MongoIdFactory;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;

import com.mongodb.client.MongoCollection;

/**
 * @author bhunt
 * 
 */
@Component(name="DefaultStreamFactory", immediate=true, service= {InputStreamFactory.class, OutputStreamFactory.class})
public class DefaultStreamFactory implements InputStreamFactory, OutputStreamFactory {
	
	private QueryEngine queryEngine;
	private ConverterService converterService;
	private volatile Map<String, MongoIdFactory> idFactories = new ConcurrentHashMap<>();
	private volatile List<InputContentHandler> handlerList = new CopyOnWriteArrayList<>();
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.mongo.OutputStreamFactory#createOutputStream(org.eclipse.emf.common.util.URI, java.util.Map, com.mongodb.client.MongoCollection, java.util.Map)
	 */
	@Override
	public OutputStream createOutputStream(URI uri, Map<?, ?> options, MongoCollection<Document> collection, Map<Object, Object> response) {
		return new MongoOutputStream(converterService, collection, uri, idFactories, options, response);
	}

	/* (non-Javadoc)
	 * @see org.gecko.emf.mongo.InputStreamFactory#createInputStream(org.eclipse.emf.common.util.URI, java.util.Map, com.mongodb.client.MongoCollection, java.util.Map)
	 */
	@Override
	public InputStream createInputStream(URI uri, Map<?, ?> options, MongoCollection<Document> collection, Map<Object, Object> response) throws IOException {
		return new MongoInputStream(converterService, queryEngine, collection, handlerList, uri, options, response);
	}

	/**
	 * Sets the converter service
	 * @param converterService the converter service to set
	 */
	@Reference(name="ConverterService", policy=ReferencePolicy.STATIC, cardinality=ReferenceCardinality.MANDATORY)
	public void setConverterService(ConverterService converterService) {
		this.converterService = converterService;
	}

	/**
	 * Sets the query engine
	 * @param queryEngine the query engine to set
	 */
	@Reference(name="QueryEngine", policy=ReferencePolicy.STATIC, cardinality=ReferenceCardinality.MANDATORY)
	public void setQueryEngine(QueryEngine queryEngine) {
		this.queryEngine = queryEngine;
	}
	
	/**
	 * Sets the id factory 
	 * @param mongoIdFactory the id factory to be added
	 */
	@Reference(name="MongoIdFactory", policy=ReferencePolicy.DYNAMIC, cardinality=ReferenceCardinality.MANDATORY, unbind="removeMongoIdFactory")
	public void addMongoIdFactory(MongoIdFactory mongoIdFactory) {
		idFactories.put(mongoIdFactory.getCollectionURI(), mongoIdFactory);
	}

	/**
	 * Un-sets the id factory 
	 * @param mongoIdFactory the id factory to be removed
	 */
	public void removeMongoIdFactory(MongoIdFactory mongoIdFactory) {
		MongoIdFactory target = idFactories.get(mongoIdFactory.getCollectionURI());
		if (mongoIdFactory == target)
			idFactories.remove(mongoIdFactory.getCollectionURI());
	}
	
	/**
	 * Sets an {@link InputContentHandler} to be used
	 * @param contentHandler the id factory to be added
	 */
	@Reference(name="InputHandler", policy=ReferencePolicy.DYNAMIC, cardinality=ReferenceCardinality.MULTIPLE, unbind="removeInputHandler")
	public void addInputHandler(InputContentHandler contentHandler) {
		handlerList.add(contentHandler);
	}
	
	/**
	 * Un-sets an {@link InputContentHandler} to be used
	 * @param contentHandler the content handler to be removed
	 */
	public void removeInputHandler(InputContentHandler contentHandler) {
		handlerList.remove(contentHandler);
	}

}
