/**
 * Copyright (c) 2012 - 2017 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.emf.osgi.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.Resource.IOWrappedException;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.emf.osgi.EMFNamespaces;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.emf.osgi.model.test.Person;
import org.gecko.emf.osgi.model.test.TestFactory;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceReference;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;

/**
 * Tests the EMF OSGi integration
 * @author Mark Hoffmann
 * @since 25.07.2017
 */
@RunWith(MockitoJUnitRunner.class)
public class DynamicModelConfiguratorTest {

	private final BundleContext context = FrameworkUtil.getBundle(DynamicModelConfiguratorTest.class).getBundleContext();
	private Configuration dynamicModelConfig;

	@Before
	public void before() {
	}

	@After
	public void after() {
		if (dynamicModelConfig != null) {
			try {
				dynamicModelConfig.delete();
			} catch (Exception e) {
				fail("Not expected that");
			}
		}
		dynamicModelConfig = null;
	}

	/**
	 * Trying to load an instance with a registered dynamic {@link EPackage}
	 * @throws IOException 
	 * @throws InterruptedException 
	 */
	@Test
	public void testCreateDynamicModel() throws IOException, InterruptedException {
		ServiceReference<ConfigurationAdmin> serviceReference = context.getServiceReference(ConfigurationAdmin.class);
		ConfigurationAdmin configAdmin = context.getService(serviceReference);
		assertNotNull(configAdmin);
		
		dynamicModelConfig = configAdmin.createFactoryConfiguration(EMFNamespaces.DYNAMIC_MODEL_CONFIGURATOR_CONFIG_NAME, "?");
		assertNotNull(dynamicModelConfig);
		ServiceReference<ResourceSetFactory> reference = context.getServiceReference(ResourceSetFactory.class);
		assertNotNull(reference);
		Object modelNames = reference.getProperty(EMFNamespaces.EMF_MODEL_NAME);
		assertNotNull(modelNames);
		assertTrue(modelNames instanceof String[]);
		List<String> modelNameList = Arrays.asList((String[]) modelNames);
		assertTrue(modelNameList.contains("ecore"));
		assertFalse(modelNameList.contains("test"));
		
		Dictionary<String,Object> properties = dynamicModelConfig.getProperties();
		assertNull(properties);
		properties = new Hashtable<>();
		properties.put(EMFNamespaces.EMF_MODEL_NAME, "test");
		properties.put(EMFNamespaces.PROP_DYNAMIC_CONFIG_CONTENT_TYPE,"test#1.0");
		properties.put(EMFNamespaces.PROP_DYNAMIC_CONFIG_FILE_EXTENSION,"test");
		properties.put(EMFNamespaces.PROP_DYNAMIC_CONFIG_ECORE_PATH,"org.gecko.emf.osgi.test.model/test.ecore");
		dynamicModelConfig.update(properties);
		Thread.sleep(2000l);
		
		reference = context.getServiceReference(ResourceSetFactory.class);
		assertNotNull(reference);
		modelNames = reference.getProperty(EMFNamespaces.EMF_MODEL_NAME);
		assertNotNull(modelNames);
		assertTrue(modelNames instanceof String[]);
		modelNameList = Arrays.asList((String[]) modelNames);
		assertTrue(modelNameList.contains("ecore"));
		assertTrue(modelNameList.contains("test"));
		
		ResourceSetFactory factory = context.getService(reference);
		assertNotNull(factory);
		ResourceSet rs = factory.createResourceSet();
		assertNotNull(rs);
		URI uri = URI.createURI("person.test");
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		Resource testSaveResource = rs.createResource(uri);
		assertNotNull(testSaveResource);
		Person p = TestFactory.eINSTANCE.createPerson();
		p.setFirstName("Emil");
		p.setLastName("Tester");
		testSaveResource.getContents().add(p);
		testSaveResource.save(baos, null);
		
		byte[] content = baos.toByteArray();
		ByteArrayInputStream bais = new ByteArrayInputStream(content);
		
		Resource testLoadResource = rs.createResource(uri);
		testLoadResource.load(bais, null);
		assertFalse(testLoadResource.getContents().isEmpty());
		EObject eo = testLoadResource.getContents().get(0);
		assertFalse(eo instanceof Person);
		EStructuralFeature firstName = eo.eClass().getEStructuralFeature("firstName");
		assertNotNull(firstName);
		EStructuralFeature lastName = eo.eClass().getEStructuralFeature("lastName");
		assertNotNull(lastName);
		assertEquals("Emil", eo.eGet(firstName));
		assertEquals("Tester", eo.eGet(lastName));
	}
	
	/**
	 * Trying to load an instance with a registered {@link EPackage}. Unregister the dynamic package and try to load the resource again, 
	 * which may fail
	 * @throws IOException 
	 * @throws InterruptedException 
	 */
	@Test(expected=IOWrappedException.class)
	public void testCreateDynamicModelUnregister() throws IOException, InterruptedException {
		ServiceReference<ConfigurationAdmin> serviceReference = context.getServiceReference(ConfigurationAdmin.class);
		ConfigurationAdmin configAdmin = context.getService(serviceReference);
		assertNotNull(configAdmin);
		
		dynamicModelConfig = configAdmin.createFactoryConfiguration(EMFNamespaces.DYNAMIC_MODEL_CONFIGURATOR_CONFIG_NAME, "?");
		assertNotNull(dynamicModelConfig);
		ServiceReference<ResourceSetFactory> reference = context.getServiceReference(ResourceSetFactory.class);
		assertNotNull(reference);
		Object modelNames = reference.getProperty(EMFNamespaces.EMF_MODEL_NAME);
		assertNotNull(modelNames);
		assertTrue(modelNames instanceof String[]);
		List<String> modelNameList = Arrays.asList((String[]) modelNames);
		assertTrue(modelNameList.contains("ecore"));
		assertFalse(modelNameList.contains("test"));
		
		Dictionary<String,Object> properties = dynamicModelConfig.getProperties();
		assertNull(properties);
		properties = new Hashtable<>();
		properties.put(EMFNamespaces.EMF_MODEL_NAME, "test");
		properties.put(EMFNamespaces.PROP_DYNAMIC_CONFIG_CONTENT_TYPE,"test#1.0");
		properties.put(EMFNamespaces.PROP_DYNAMIC_CONFIG_FILE_EXTENSION,"test");
		properties.put(EMFNamespaces.PROP_DYNAMIC_CONFIG_ECORE_PATH,"org.gecko.emf.osgi.test.model/test.ecore");
		dynamicModelConfig.update(properties);
		Thread.sleep(2000l);
		
		reference = context.getServiceReference(ResourceSetFactory.class);
		assertNotNull(reference);
		modelNames = reference.getProperty(EMFNamespaces.EMF_MODEL_NAME);
		assertNotNull(modelNames);
		assertTrue(modelNames instanceof String[]);
		modelNameList = Arrays.asList((String[]) modelNames);
		assertTrue(modelNameList.contains("ecore"));
		assertTrue(modelNameList.contains("test"));
		
		ResourceSetFactory factory = context.getService(reference);
		assertNotNull(factory);
		ResourceSet rs = factory.createResourceSet();
		assertNotNull(rs);
		URI uri = URI.createURI("person.test");
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		Resource testSaveResource = rs.createResource(uri);
		assertNotNull(testSaveResource);
		Person p = TestFactory.eINSTANCE.createPerson();
		p.setFirstName("Emil");
		p.setLastName("Tester");
		testSaveResource.getContents().add(p);
		testSaveResource.save(baos, null);
		
		byte[] content = baos.toByteArray();
		ByteArrayInputStream bais = new ByteArrayInputStream(content);
		
		Resource testLoadResource = rs.createResource(uri);
		testLoadResource.load(bais, null);
		assertFalse(testLoadResource.getContents().isEmpty());
		EObject eo = testLoadResource.getContents().get(0);
		assertFalse(eo instanceof Person);
		EStructuralFeature firstName = eo.eClass().getEStructuralFeature("firstName");
		assertNotNull(firstName);
		EStructuralFeature lastName = eo.eClass().getEStructuralFeature("lastName");
		assertNotNull(lastName);
		assertEquals("Emil", eo.eGet(firstName));
		assertEquals("Tester", eo.eGet(lastName));
		
		dynamicModelConfig.delete();
		dynamicModelConfig = null;
		
		Thread.sleep(2000l);
		testLoadResource = rs.createResource(uri);
		testLoadResource.load(bais, null);
	}
	
}