/**
 * Copyright (c) 2012 - 2017 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.emf.osgi.tests;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;

import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.Resource.Factory.Registry;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.emf.osgi.EMFNamespaces;
import org.gecko.emf.osgi.EPackageConfigurator;
import org.gecko.emf.osgi.ResourceFactoryConfigurator;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.emf.osgi.model.test.TestPackage;
import org.gecko.emf.osgi.model.test.configurator.TestPackageConfigurator;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;

/**
 * Integration test for the {@link ResourceSetFactory}
 * @author Mark Hoffmann
 * @since 25.07.2017
 */
@RunWith(MockitoJUnitRunner.class)
public class IsolatedResourceSetFactoryIntegrationTest {

	private final BundleContext context = FrameworkUtil.getBundle(IsolatedResourceSetFactoryIntegrationTest.class).getBundleContext();
	private Configuration configuration;
	private ServiceRegistration<?> epackageRegistration1;
	private ServiceRegistration<?> epackageRegistration2;
	
	@Before
	public void before() {
	}

	@After
	public void after() {
		if (configuration != null) {
			try {
				configuration.delete();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		if (epackageRegistration1 != null) {
			epackageRegistration1.unregister();
			epackageRegistration1 = null;
		}
		if (epackageRegistration2 != null) {
			epackageRegistration2.unregister();
			epackageRegistration2 = null;
		}
	}

	/**
	 * Tests, if the service was set up correctly
	 * @throws IOException 
	 * @throws InterruptedException 
	 * @throws InvalidSyntaxException 
	 */
	@Test
	public void testResourceSetFactoryExists() throws IOException, InterruptedException, InvalidSyntaxException {
		ServiceReference<ConfigurationAdmin> serviceReference = context.getServiceReference(ConfigurationAdmin.class);
		ConfigurationAdmin configAdmin = context.getService(serviceReference);
		assertNotNull(configAdmin);
		Configuration config = configAdmin.createFactoryConfiguration(EMFNamespaces.ISOLATED_RESOURCE_SET_FACTORY_CONFIG_NAME, "?");
		assertNotNull(config);
		Dictionary<String,Object> properties = config.getProperties();
		assertNull(properties);
		properties = new Hashtable<>();
		properties.put("rsf.name", "test");
		properties.put("rsf.model.target.filter", "(" + EMFNamespaces.EMF_MODEL_NAME + "=*)");
		config.update(properties);
		Thread.sleep(3000l);
		Collection<ServiceReference<Registry>> rfrRef = context.getServiceReferences(Resource.Factory.Registry.class, "(rsf.name=test)");
		assertFalse(rfrRef.isEmpty());
		Collection<ServiceReference<org.eclipse.emf.ecore.EPackage.Registry>> eprRef = context.getServiceReferences(EPackage.Registry.class, "(rsf.name=test)");
		assertFalse(eprRef.isEmpty());
		Collection<ServiceReference<ResourceSetFactory>> rsfRef = context.getServiceReferences(ResourceSetFactory.class, "(rsf.name=test)");
		assertFalse(rsfRef.isEmpty());
	}
	
	/**
	 * Tests, if the service was set up correctly
	 * @throws IOException 
	 * @throws InterruptedException 
	 * @throws InvalidSyntaxException 
	 */
	@Test
	public void testResourceSetExists() throws IOException, InterruptedException, InvalidSyntaxException {
		ServiceReference<ConfigurationAdmin> serviceReference = context.getServiceReference(ConfigurationAdmin.class);
		ConfigurationAdmin configAdmin = context.getService(serviceReference);
		assertNotNull(configAdmin);
		Configuration config = configAdmin.createFactoryConfiguration(EMFNamespaces.ISOLATED_RESOURCE_SET_FACTORY_CONFIG_NAME, "?");
		assertNotNull(config);
		Dictionary<String,Object> properties = config.getProperties();
		assertNull(properties);
		properties = new Hashtable<>();
		properties.put("rsf.name", "test");
		properties.put("rsf.model.target.filter", "(" + EMFNamespaces.EMF_MODEL_NAME + "=*)");
		config.update(properties);
		Thread.sleep(3000l);
		Collection<ServiceReference<Registry>> rfrRef = context.getServiceReferences(Resource.Factory.Registry.class, "(rsf.name=test)");
		assertFalse(rfrRef.isEmpty());
		Collection<ServiceReference<org.eclipse.emf.ecore.EPackage.Registry>> eprRef = context.getServiceReferences(EPackage.Registry.class, "(rsf.name=test)");
		assertFalse(eprRef.isEmpty());
		Collection<ServiceReference<ResourceSet>> rsRef = context.getServiceReferences(ResourceSet.class, "(rsf.name=test)");
		assertFalse(rsRef.isEmpty());
	}
	
	/**
	 * Tests, if the service was set up correctly
	 * @throws IOException 
	 * @throws InterruptedException 
	 * @throws InvalidSyntaxException 
	 */
	@Test
	public void testResourceSetFactoryRegister() throws IOException, InterruptedException, InvalidSyntaxException {
		ServiceReference<ConfigurationAdmin> serviceReference = context.getServiceReference(ConfigurationAdmin.class);
		ConfigurationAdmin configAdmin = context.getService(serviceReference);
		assertNotNull(configAdmin);
		Configuration config = configAdmin.createFactoryConfiguration(EMFNamespaces.ISOLATED_RESOURCE_SET_FACTORY_CONFIG_NAME, "?");
		assertNotNull(config);
		Dictionary<String,Object> properties = config.getProperties();
		assertNull(properties);
		properties = new Hashtable<>();
		properties.put("rsf.name", "test");
		properties.put("rsf.model.target.filter", "(" + EMFNamespaces.EMF_MODEL_NAME + "=test)");
		config.update(properties);
		Thread.sleep(3000l);
		Collection<ServiceReference<Registry>> rfrRefs = context.getServiceReferences(Resource.Factory.Registry.class, "(rsf.name=test)");
		assertFalse(rfrRefs.isEmpty());
		Collection<ServiceReference<org.eclipse.emf.ecore.EPackage.Registry>> eprRefs = context.getServiceReferences(EPackage.Registry.class, "(rsf.name=test)");
		assertFalse(eprRefs.isEmpty());
		Collection<ServiceReference<ResourceSetFactory>> rsfRefs = context.getServiceReferences(ResourceSetFactory.class, "(rsf.name=test)");
		assertFalse(rsfRefs.isEmpty());
		
		ServiceReference<ResourceSetFactory> rsfRef = rsfRefs.iterator().next();
		Object modelNames = rsfRef.getProperty(EMFNamespaces.EMF_MODEL_NAME);
		assertNotNull(modelNames);
		assertTrue(modelNames instanceof String[]);
		List<String> modelNameList = Arrays.asList((String[]) modelNames);
		assertTrue(modelNameList.contains("ecore"));
		assertFalse(modelNameList.contains("test"));
		assertFalse(modelNameList.contains("test2"));
		
		Dictionary<String, Object> epackageProperties = new Hashtable<String, Object>();
		epackageProperties.put(EMFNamespaces.EMF_MODEL_NAME, TestPackage.eNAME);
		epackageRegistration1 = context.registerService(new String[] {EPackageConfigurator.class.getName(), ResourceFactoryConfigurator.class.getName()}, new TestPackageConfigurator(), epackageProperties);
		Thread.sleep(1000l);
		
		rsfRefs = context.getServiceReferences(ResourceSetFactory.class, "(rsf.name=test)");
		assertFalse(rsfRefs.isEmpty());
		
		rsfRef = rsfRefs.iterator().next();
		modelNames = rsfRef.getProperty(EMFNamespaces.EMF_MODEL_NAME);
		assertNotNull(modelNames);
		assertTrue(modelNames instanceof String[]);
		modelNameList = Arrays.asList((String[]) modelNames);
		assertTrue(modelNameList.contains("ecore"));
		assertTrue(modelNameList.contains("test"));
		assertFalse(modelNameList.contains("test2"));
		
		epackageProperties = new Hashtable<String, Object>();
		epackageProperties.put(EMFNamespaces.EMF_MODEL_NAME, TestPackage.eNAME + "2");
		epackageRegistration2 = context.registerService(new String[] {EPackageConfigurator.class.getName(), ResourceFactoryConfigurator.class.getName()}, new TestPackageConfigurator(), epackageProperties);
		Thread.sleep(1000l);
		
		rsfRefs = context.getServiceReferences(ResourceSetFactory.class, "(rsf.name=test)");
		assertFalse(rsfRefs.isEmpty());
		
		Collection<ServiceReference<EPackageConfigurator>> testRefs = context.getServiceReferences(EPackageConfigurator.class, "(" + EMFNamespaces.EMF_MODEL_NAME + "=test2)");
		assertFalse(testRefs.isEmpty());
		
		rsfRef = rsfRefs.iterator().next();
		modelNames = rsfRef.getProperty(EMFNamespaces.EMF_MODEL_NAME);
		assertNotNull(modelNames);
		assertTrue(modelNames instanceof String[]);
		modelNameList = Arrays.asList((String[]) modelNames);
		assertTrue(modelNameList.contains("ecore"));
		assertTrue(modelNameList.contains("test"));
		assertFalse(modelNameList.contains("test2"));
		
	}
	
	/**
	 * Tests, if the service was set up correctly
	 * @throws IOException 
	 * @throws InterruptedException 
	 * @throws InvalidSyntaxException 
	 */
	@Test
	public void testResourceSetRegister() throws IOException, InterruptedException, InvalidSyntaxException {
		ServiceReference<ConfigurationAdmin> serviceReference = context.getServiceReference(ConfigurationAdmin.class);
		ConfigurationAdmin configAdmin = context.getService(serviceReference);
		assertNotNull(configAdmin);
		Configuration config = configAdmin.createFactoryConfiguration(EMFNamespaces.ISOLATED_RESOURCE_SET_FACTORY_CONFIG_NAME, "?");
		assertNotNull(config);
		Dictionary<String,Object> properties = config.getProperties();
		assertNull(properties);
		properties = new Hashtable<>();
		properties.put("rsf.name", "test");
		properties.put("rsf.model.target.filter", "(" + EMFNamespaces.EMF_MODEL_NAME + "=test)");
		config.update(properties);
		Thread.sleep(3000l);
		Collection<ServiceReference<Registry>> rfrRefs = context.getServiceReferences(Resource.Factory.Registry.class, "(rsf.name=test)");
		assertFalse(rfrRefs.isEmpty());
		Collection<ServiceReference<org.eclipse.emf.ecore.EPackage.Registry>> eprRefs = context.getServiceReferences(EPackage.Registry.class, "(rsf.name=test)");
		assertFalse(eprRefs.isEmpty());
		Collection<ServiceReference<ResourceSet>> rsRefs = context.getServiceReferences(ResourceSet.class, "(rsf.name=test)");
		assertFalse(rsRefs.isEmpty());
		
		ServiceReference<ResourceSet> rsRef = rsRefs.iterator().next();
		Object modelNames = rsRef.getProperty(EMFNamespaces.EMF_MODEL_NAME);
		assertNotNull(modelNames);
		assertTrue(modelNames instanceof String[]);
		List<String> modelNameList = Arrays.asList((String[]) modelNames);
		assertTrue(modelNameList.contains("ecore"));
		assertFalse(modelNameList.contains("test"));
		assertFalse(modelNameList.contains("test2"));
		
		Dictionary<String, Object> epackageProperties = new Hashtable<String, Object>();
		epackageProperties.put(EMFNamespaces.EMF_MODEL_NAME, TestPackage.eNAME);
		epackageRegistration1 = context.registerService(new String[] {EPackageConfigurator.class.getName(), ResourceFactoryConfigurator.class.getName()}, new TestPackageConfigurator(), epackageProperties);
		Thread.sleep(1000l);
		
		rsRefs = context.getServiceReferences(ResourceSet.class, "(rsf.name=test)");
		assertFalse(rsRefs.isEmpty());
		
		rsRef = rsRefs.iterator().next();
		modelNames = rsRef.getProperty(EMFNamespaces.EMF_MODEL_NAME);
		assertNotNull(modelNames);
		assertTrue(modelNames instanceof String[]);
		modelNameList = Arrays.asList((String[]) modelNames);
		assertTrue(modelNameList.contains("ecore"));
		assertTrue(modelNameList.contains("test"));
		assertFalse(modelNameList.contains("test2"));
		
		epackageProperties = new Hashtable<String, Object>();
		epackageProperties.put(EMFNamespaces.EMF_MODEL_NAME, TestPackage.eNAME + "2");
		epackageRegistration2 = context.registerService(new String[] {EPackageConfigurator.class.getName(), ResourceFactoryConfigurator.class.getName()}, new TestPackageConfigurator(), epackageProperties);
		Thread.sleep(1000l);
		
		rsRefs = context.getServiceReferences(ResourceSet.class, "(rsf.name=test)");
		assertFalse(rsRefs.isEmpty());
		
		Collection<ServiceReference<EPackageConfigurator>> testRefs = context.getServiceReferences(EPackageConfigurator.class, "(" + EMFNamespaces.EMF_MODEL_NAME + "=test2)");
		assertFalse(testRefs.isEmpty());
		
		rsRef = rsRefs.iterator().next();
		modelNames = rsRef.getProperty(EMFNamespaces.EMF_MODEL_NAME);
		assertNotNull(modelNames);
		assertTrue(modelNames instanceof String[]);
		modelNameList = Arrays.asList((String[]) modelNames);
		assertTrue(modelNameList.contains("ecore"));
		assertTrue(modelNameList.contains("test"));
		assertFalse(modelNameList.contains("test2"));
		
	}
	
}