/**
 * Copyright (c) 2012 - 2017 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.mongo.osgi.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import java.io.IOException;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.gecko.mongo.osgi.MongoClientProvider;
import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.gecko.mongo.osgi.configuration.ConfigurationProperties;
import org.gecko.mongo.osgi.tests.customizer.MongoProviderCustomizer;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.util.tracker.ServiceTracker;

import com.mongodb.client.MongoDatabase;

/**
 * Tests the database provider
 * @author Mark Hoffmann
 * @since 25.07.2017
 */
@RunWith(MockitoJUnitRunner.class)
public class MongoDatabaseProviderTest {

	private final BundleContext context = FrameworkUtil.getBundle(MongoDatabaseProviderTest.class).getBundleContext();

	private String mongoHost = System.getProperty("mongo.host", "localhost");
	
	@Before
	public void setUp() throws Exception {
	}

	@After
	public void tearDown() throws Exception {
	}

	@Test
	public void testNoMongoClientProvider() throws InvalidSyntaxException, BundleException, IOException, InterruptedException {

		// service lookup for configuration admin service
		ServiceReference<ConfigurationAdmin> caRef = context.getServiceReference(ConfigurationAdmin.class);
		assertNotNull(caRef);
		ConfigurationAdmin cm = context.getService(caRef);
		assertNotNull(cm);

		// create mongo client configuration
		Configuration databaseConfig = cm.getConfiguration(ConfigurationProperties.DATABASE_PID, "?");
		assertNotNull(databaseConfig);

		// has to be a new configuration
		Dictionary<String, Object> p = databaseConfig.getProperties();
		assertNull(p);

		// add service properties
		String dbAlias = "testDB";
		String db = "test";
		p = new Hashtable<String, Object>();
		p.put(MongoDatabaseProvider.PROP_ALIAS, dbAlias);
		p.put(MongoDatabaseProvider.PROP_DATABASE, db);
		databaseConfig.update(p);

		// re-check configuration must not available, because of the missing client
		databaseConfig = cm.getConfiguration(ConfigurationProperties.CLIENT_PID, "?");
		assertNull(databaseConfig.getProperties());

		databaseConfig.delete();

	}

	@Test
	public void testCreateMongoDatabaseProvider() throws InvalidSyntaxException, BundleException, IOException, InterruptedException {

		// service lookup for configuration admin service
		ServiceReference<ConfigurationAdmin> caRef = context.getServiceReference(ConfigurationAdmin.class);
		assertNotNull(caRef);
		ConfigurationAdmin cm = context.getService(caRef);
		assertNotNull(cm);
		Configuration clientConfig = cm.getConfiguration(ConfigurationProperties.CLIENT_PID, "?");
		assertNotNull(clientConfig);

		// has to be a new configuration
		Dictionary<String, Object> p = clientConfig.getProperties();
		assertNull(p);

		// add service properties
		String clientId = "testClient";
		String clientUri = "mongodb://" + mongoHost + ":27017";
		p = new Hashtable<String, Object>();
		p.put(MongoClientProvider.PROP_CLIENT_ID, clientId);
		p.put(MongoClientProvider.PROP_URI, clientUri);
		clientConfig.update(p);

		// now track the MongoClientProvider service
		final CountDownLatch createLatch = new CountDownLatch(1);
		MongoProviderCustomizer<MongoClientProvider, MongoClientProvider> customizer = new MongoProviderCustomizer<MongoClientProvider, MongoClientProvider>(context, createLatch);
		ServiceTracker<MongoClientProvider, MongoClientProvider> tracker = new ServiceTracker<MongoClientProvider, MongoClientProvider>(context, MongoClientProvider.class, customizer);
		tracker.open(true);

		// wait maximum 5 second until then the service must be created
		createLatch.await(5, TimeUnit.SECONDS);
		assertEquals(0, createLatch.getCount());

		// create data base provider configuration
		Configuration databaseConfig = cm.getConfiguration(ConfigurationProperties.DATABASE_PID, "?");
		assertNotNull(databaseConfig);

		// add service properties
		String dbAlias = "testDB";
		String db = "test";
		Dictionary<String, Object> dbp = new Hashtable<String, Object>();
		dbp.put(MongoDatabaseProvider.PROP_ALIAS, dbAlias);
		dbp.put(MongoDatabaseProvider.PROP_DATABASE, db);
		databaseConfig.update(dbp);

		// now track the MongoDatabaseProvider service
		CountDownLatch dbCreateLatch = new CountDownLatch(1);
		MongoProviderCustomizer<MongoDatabaseProvider, MongoDatabaseProvider> dbCustomizer = new MongoProviderCustomizer<MongoDatabaseProvider, MongoDatabaseProvider>(context, dbCreateLatch);
		ServiceTracker<MongoDatabaseProvider, MongoDatabaseProvider> dbTracker = new ServiceTracker<MongoDatabaseProvider, MongoDatabaseProvider>(context, MongoDatabaseProvider.class, dbCustomizer);
		dbTracker.open(true);

		// wait maximum 5 second until then the service must be created
		dbCreateLatch.await(5, TimeUnit.SECONDS);
		assertEquals(0, dbCreateLatch.getCount());

		// get the MongoDatabaseProvider
		ServiceReference<?>[] databaseRefs = context.getAllServiceReferences(MongoDatabaseProvider.class.getName(), null);
		assertNotNull(databaseRefs);
		assertEquals(1, databaseRefs.length);
		ServiceReference<?> databaseRef = databaseRefs[0];
		MongoDatabaseProvider databaseProvider = (MongoDatabaseProvider) context.getService(databaseRef);
		MongoDatabase database = databaseProvider.getDatabase();
		assertEquals(db, database.getName());

		databaseConfig.delete();
		clientConfig.delete();
	}

}
