/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.adapter.amqp;

import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.gecko.adapter.amqp.client.AMQPContext;
import org.gecko.adapter.amqp.client.AMQPContextBuilder;
import org.gecko.adapter.amqp.consumer.AMQPConsumer;
import org.gecko.osgi.messaging.Message;
//import org.gecko.osgi.messaging.MessagingConstants;
import org.gecko.osgi.messaging.MessagingContext;
import org.gecko.osgi.messaging.MessagingService;
import org.gecko.osgi.messaging.PushStreamHelper;
//import org.osgi.annotation.bundle.Capability;
import org.osgi.framework.BundleContext;
import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;
import org.osgi.util.pushstream.PushEvent;
import org.osgi.util.pushstream.PushStream;
import org.osgi.util.pushstream.PushStreamBuilder;
import org.osgi.util.pushstream.PushStreamProvider;
import org.osgi.util.pushstream.SimplePushEventSource;

import com.rabbitmq.client.AMQP;
import com.rabbitmq.client.Channel;
import com.rabbitmq.client.Connection;
import com.rabbitmq.client.ConnectionFactory;
import com.rabbitmq.client.DefaultConsumer;
import com.rabbitmq.client.Envelope;

/**
 * Implementation of the messaging service for the AMQP protocol, using the RabbitMQ AMQP client
 * @see https://www.rabbitmq.com/api-guide.html
 * @author Mark Hoffmann
 * @since 15.02.2018
 */
//@Capability(namespace=MessagingConstants.CAPABILITY_NAMESPACE, name="amqp.adapter", version="1.0.0", attribute= {"vendor=Gecko.io", "implementation=RabbitMQ"})
@Component(service=MessagingService.class, name="AMQPService", configurationPolicy=ConfigurationPolicy.REQUIRE, immediate=true)
public class AMQPService implements MessagingService, AutoCloseable {

	private static final Logger logger = Logger.getLogger("o.g.a.amqpService");
	private PushStreamProvider provider = new PushStreamProvider();
	private Map<String, AMQPConsumer> consumerMap = new ConcurrentHashMap<>();
	private AtomicReference<Connection> connectionRef = new AtomicReference<Connection>();
	private Map<String, Channel> channelMap = new ConcurrentHashMap<String, Channel>();
	private ConnectionFactory connectionFactory;

	@ObjectClassDefinition
	@interface AMQPConfig {

		String username();
		String password();
		String host() default "localhost";
		int port() default 5672;
		String virtualHost() default "";
		boolean autoRecovery() default false;

		String brokerUrl();

	}		

	@Activate	
	void activate(AMQPConfig config, BundleContext context) throws Exception {
		try {
			connectionFactory = configureConnectionFactory(config);
			connect();
		} catch(Exception e){
			logger.log(Level.SEVERE, "Error creating AMQP connection", e);
			throw e;
		}
	}

	/**
	 * Called on component deactivation
	 * @throws Exception
	 */
	@Deactivate
	void deactivate() throws Exception {
		close();
	}
	/* 
	 * (non-Javadoc)
	 * @see java.lang.AutoCloseable#close()
	 */
	@Override
	public void close() throws Exception {
		channelMap.keySet().forEach((k)->{
			try {
				disconnectChannel(k);
			} catch (Exception e) {
				logger.log(Level.SEVERE, "Error closing channel: " + k, e);
			}
		});
		channelMap.clear();
		Connection connection = connectionRef.get();
		if (connection != null) {
			try {
				connection.close();
			} catch (Exception e) {
				logger.log(Level.SEVERE, "Error closing connection ", e);
			}
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.osgi.messaging.MessagingService#publish(java.lang.String, java.nio.ByteBuffer)
	 */
	@Override
	public void publish(String topic, ByteBuffer content) throws Exception {
		MessagingContext context = new AMQPContextBuilder().topic().durable().queue(topic).build();
		publish(topic, content, context);
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.osgi.messaging.MessagingService#publish(java.lang.String, java.nio.ByteBuffer, org.gecko.osgi.messaging.MessagingContext)
	 */
	@Override
	public void publish(String topic, ByteBuffer content, MessagingContext context) throws Exception {
		Channel channel = null;
		if (context != null && context instanceof AMQPContext) {
			AMQPContext ctx = (AMQPContext) context;
			ctx.setQueueName(topic);
			channel = ctx.isExchangeMode() ? connectExchange(ctx) : connectQueue(ctx);
			if (channel.isOpen()) {
				byte[] message = content.array();
				if (ctx.isExchangeMode()) {
					channel.basicPublish(ctx.getExchangeName(), ctx.getRoutingKey(), null, message);
				} else {
					channel.basicPublish("", ctx.getQueueName(), null, message);
				}
			}
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.osgi.messaging.MessagingService#subscribe(java.lang.String)
	 */
	@Override
	public PushStream<Message> subscribe(String topic) throws Exception {
		MessagingContext context = new AMQPContextBuilder().topic().queue(topic).durable().build();
		return subscribe(topic, context);
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.osgi.messaging.MessagingService#subscribe(java.lang.String, org.gecko.osgi.messaging.MessagingContext)
	 */
	@Override
	public PushStream<Message> subscribe(String topic, MessagingContext context) throws Exception {
		if (context != null && context instanceof AMQPContext) {
			AMQPContext ctx = (AMQPContext) context;
			AMQPConsumer consumer = consumerMap.get(topic); 
			String consumerTag = "ma_" + topic;
			Channel channel = connectQueue(ctx);
			SimplePushEventSource<Message> source = null;
			if (channel.isOpen()) {
				if (consumer == null) {
					source = provider.buildSimpleEventSource(Message.class).build();
					consumer = new AMQPConsumer(channel, source, topic);
					consumerMap.put(topic, consumer);
				}
				channel.basicConsume(ctx.getQueueName(), ctx.isAutoAcknowledge(), consumerTag, consumer);
				PushStreamBuilder<Message,BlockingQueue<PushEvent<? extends Message>>> buildStream = PushStreamHelper.buildPushStreamWithContext(context, source);
				return buildStream.build();
			} else {
				throw new IllegalStateException("The channel to connect is not open");
			}
		} else {
			throw new IllegalArgumentException("The message context is not of type AMQPContext");
		}
	}

	/**
	 * Configures the {@link ConnectionFactory}
	 * @param config the configuration admin configuration
	 * @return the configures connection factory instance
	 * @throws ConfigurationException
	 */
	private ConnectionFactory configureConnectionFactory(AMQPConfig config) throws ConfigurationException {
		if (config == null) {
			throw new IllegalArgumentException("Cannot create a connection factory without a configuration");
		}
		boolean useUrl = false;
		ConnectionFactory conFactory = new ConnectionFactory();
		if (config.brokerUrl() != null && config.brokerUrl().startsWith("amqp://")) {
			try {
				conFactory.setUri(config.brokerUrl());
				useUrl = true;
			} catch (KeyManagementException | NoSuchAlgorithmException | URISyntaxException e) {
				logger.log(Level.SEVERE, "Error setting the URI to connection factroy " + config.brokerUrl(), e);
			}
		} 
		if (!useUrl && validateConfiguration(config)) {
			conFactory.setPort(config.port());
			conFactory.setHost(config.host());
			conFactory.setVirtualHost(config.virtualHost());
			if (config.username() != null && !config.username().isEmpty()) {
				conFactory.setUsername(config.username());
			}
			if (config.password() != null && !config.password().isEmpty()) {
				conFactory.setPassword(config.password());
			}
		} else {
			if (!useUrl) {
				throw new ConfigurationException("amqp.configuration", "Error validating AMQP configuration, there are missing mandatory values");
			}
		}
		if (config.autoRecovery()) {
			conFactory.setAutomaticRecoveryEnabled(config.autoRecovery());
		}
		return conFactory;
	}

	/**
	 * Validates the configuration and returns <code>true</code>, if mandatory values are valid
	 * @param config the configuration annotation
	 * @return <code>true</code>, if mandatory values are valid
	 */
	private boolean validateConfiguration(AMQPConfig config) {
		return config != null & 
				config.port() > 0 && 
				config.host() != null && !config.host().isEmpty() && 
				config.virtualHost() != null && !config.virtualHost().isEmpty();
	}

	/**
	 * Creates a new connection
	 * @throws IOException
	 * @throws TimeoutException
	 */
	private void connect() throws IOException, TimeoutException {
		Connection connection = connectionRef.get();
		if (connection == null || 
				!connection.isOpen()) {
			connectionRef.set(connectionFactory.newConnection());
		}
	}

	/**
	 * Disconnects a channel with the given key
	 * @param key
	 * @throws IOException
	 * @throws TimeoutException
	 */
	private void disconnectChannel(String key) throws IOException, TimeoutException {
		Channel channel = channelMap.get(key);
		if (channel == null) {
			logger.warning("No channel exists for key: " + key + " - Nothing to disconnect");
			return;
		}
		channel.close();
	}

	/**
	 * Connects using an exchange and routing key to a routing type
	 * @param context the context object
	 * @return the channel instance
	 * @throws IOException
	 * @throws TimeoutException
	 */
	private Channel connectExchange(AMQPContext context) throws IOException, TimeoutException {
		String exchange = context.getExchangeName();
		String routingKey = context.getRoutingKey();
		String routingType = context.getRoutingType();
		String key = exchange + "_" + routingKey + "_" + routingType;
		if (channelMap.containsKey(key)) {
			logger.info("Channel already exists for exchange: " + exchange + " and routing key: " + routingKey + " and type: " + routingType);
			return channelMap.get(key);
		}
		connect();
		Channel channel = connectionRef.get().createChannel();
		channel.exchangeDeclare(exchange, routingType, context.isDurable(), context.isAutoDelete(), null);
		String queueName = channel.queueDeclare().getQueue();
		channel.queueBind(queueName, exchange, routingKey);
		channelMap.put(key, channel);
		logger.info("Created channel for type '" + routingType + "', exchange: " + exchange + " and routing key: " + routingKey);
		return channel;
	}

	/**
	 * Connects using a queue name
	 * @param context the context object
	 * @return the channel instance
	 * @throws IOException
	 * @throws TimeoutException
	 */
	private Channel connectQueue(AMQPContext context) throws IOException, TimeoutException {
		String queue = context.getQueueName();
		if (channelMap.containsKey(queue)) {
			logger.info("Channel already created for queue: " + queue);
			return channelMap.get(queue);
		}
		connect();
		Channel channel = connectionRef.get().createChannel();
		channel.queueDeclare(queue , context.isDurable(), context.isExclusive(), context.isAutoDelete(), null);
		channelMap.put(queue, channel);
		logger.info("Created channel for queue: " + queue);
		return channel;
	}

	/**
	 * @param queue
	 * @param autoAck
	 * @param consumerTag
	 * @param consumer
	 * @throws IOException 
	 * @throws TimeoutException 
	 */
	public void registerConsumerQueue(String queue, boolean autoAck, String consumerTag, Consumer<byte[]> consumer) throws IOException, TimeoutException {
		AMQPContextBuilder ctxBuilder = new AMQPContextBuilder().topic().queue(queue);
		if (autoAck) {
			ctxBuilder.autoAcknowledge();
		}
		AMQPContext context = (AMQPContext) ctxBuilder.build();
		Channel channel = connectQueue(context);
		if (channel.isOpen()) {
			channel.basicConsume(queue, autoAck, consumerTag, new DefaultConsumer(channel) {
				@Override
				public void handleDelivery(String consumerTag, Envelope envelope, AMQP.BasicProperties properties, byte[] body)	throws IOException {
					String routingKey = envelope.getRoutingKey();
					String contentType = properties.getContentType();
					long deliveryTag = envelope.getDeliveryTag();
					logger.log(Level.INFO, "Received message: '" + new String(body) + "' with routingKey: " + routingKey + ", contentType: " + contentType + ", deliveryTag: " + deliveryTag);
					consumer.accept(body);
					channel.basicAck(deliveryTag, false);
				}
			});
		}
	}
}
