/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.adapter.amqp.consumer;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.gecko.osgi.messaging.Message;
import org.gecko.osgi.messaging.SimpleMessage;
import org.osgi.util.pushstream.SimplePushEventSource;

import com.rabbitmq.client.Channel;
import com.rabbitmq.client.DefaultConsumer;
import com.rabbitmq.client.Envelope;
import com.rabbitmq.client.AMQP.BasicProperties;

/**
 * Consumer class
 * @author Mark Hoffmann
 * @since 20.02.2018
 */
public class AMQPConsumer extends DefaultConsumer {

	private static final Logger logger = Logger.getLogger("o.g.a.amqpConsumer");
	private final SimplePushEventSource<Message> eventSource;
	private final String topic;

	/**
	 * Creates a new instance.
	 * @param channel
	 */
	public AMQPConsumer(Channel channel, SimplePushEventSource<Message> eventSource, String topic) {
		super(channel);
		this.eventSource = eventSource;
		this.topic = topic;
	}
	
	/**
	 * Returns the eventSource.
	 * @return the eventSource
	 */
	public SimplePushEventSource<Message> getEventSource() {
		return eventSource;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.rabbitmq.client.DefaultConsumer#handleDelivery(java.lang.String, com.rabbitmq.client.Envelope, com.rabbitmq.client.AMQP.BasicProperties, byte[])
	 */
	@Override
	public void handleDelivery(String consumerTag, Envelope envelope, BasicProperties properties, byte[] body)
			throws IOException {
		String routingKey = envelope.getRoutingKey();
		String contentType = properties.getContentType();
		long deliveryTag = envelope.getDeliveryTag();
		logger.log(Level.FINE, "Received message: '" + new String(body) + "' with routingKey: " + routingKey + ", contentType: " + contentType + ", deliveryTag: " + deliveryTag);
		if(eventSource.isConnected()){
			try {
				Message msg = new SimpleMessage(topic, ByteBuffer.wrap(body));
				eventSource.publish(msg);
			} catch(Exception ex){
				logger.log(Level.SEVERE, "Detected error onWebSocketText", ex);
			}
		}
		getChannel().basicAck(deliveryTag, false);
	}

}
