/*************************************************************************************
 * Copyright (c) 2005, 2020 TragWerk Software Döking+Purtak GbR, Dresden, Germany
 * Author: Fuchs
 * All rights reserved.
 *************************************************************************************/
package org.gecko.minio;

import java.io.InputStream;

/**
 * Interface for an Min.io client driver.
 *
 * @author Mark Hoffmann
 * @since 13.05.2019
 */
public interface MinioClient {

    /** The Minio Service Component Configuration PID */
    public static final String CONFIGURATION_PID = "MinioClient";

    /** The Key for the Configuration Property of the MinioConfigName - Expected Value is String */
    public static final String CONFIG_KEY_NAME = "name";

    /** The Key for the Configuration Property of the Minio Server Protocol - Expected Value is String, e.g. <code>http</code> or <code>https</code> */
    public static final String CONFIG_KEY_PROTOCOL = "protocol";

    /** The Key for the Configuration Property of the Minio Server Hostname - Expected Value is String */
    public static final String CONFIG_KEY_HOST = "hostname";

    /** The Key for the Configuration Property of the Minio Server Port - Expected Value is int */
    public static final String CONFIG_KEY_PORT = "port";

    /** The Key for the Configuration Property of the Minio Server Access Key - Expected Value is String */
    public static final String CONFIG_KEY_ACCESSKEY = "accessKey";

    /** The Key for the Configuration Property of the Minio Server Secret Key - Expected Value is String */
    public static final String CONFIG_KEY_SECRETKEY = "secretKey";

    /** The Key for the Configuration Property of the Minio Server Timeout in milliseconds - Expected Value is int */
    public static final String CONFIG_KEY_TIMEOUT = "timeout";

    /**
     * Returns the driver id
     *
     * @return the driver id
     */
    public String getId();

    /**
     * Upload file.
     *
     * @param bucketPath the bucket path
     * @param contentType the content type
     * @param input the input
     * @throws MinioClientException the minio client exception
     */
    public void uploadFile(String bucketPath, String contentType, InputStream input) throws MinioClientException;

    /**
     * Upload file.
     *
     * @param bucket the bucket
     * @param file the file
     * @param contentType the content type
     * @param input the input
     * @throws MinioClientException the minio client exception
     */
    public void uploadFile(String bucket, String file, String contentType, InputStream input) throws MinioClientException;

    /**
     * Gets the file.
     *
     * @param bucketPath the bucket path
     * @param contentType the content type
     * @return the file
     * @throws MinioClientException the minio client exception
     */
    public InputStream getFile(String bucketPath, String contentType) throws MinioClientException;

    /**
     * Gets the file.
     *
     * @param bucket the bucket
     * @param file the file
     * @param contentType the content type
     * @return the file
     * @throws MinioClientException the minio client exception
     */
    public InputStream getFile(String bucket, String file, String contentType) throws MinioClientException;

    /**
     * Delete file.
     *
     * @param bucketPath the bucket path
     * @throws MinioClientException the minio client exception
     */
    public void deleteFile(String bucketPath) throws MinioClientException;

    /**
     * Delete file.
     *
     * @param bucket the bucket
     * @param file the file
     * @throws MinioClientException the minio client exception
     */
    public void deleteFile(String bucket, String file) throws MinioClientException;

}
