/**
 * Copyright (c) 2012 - 2019 Data In Motion and others.
 * All rights reserved.
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * Data In Motion - initial API and implementation
 */
package org.gecko.minio.impl;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Date;

import org.gecko.minio.MinioClient;
import org.gecko.minio.MinioClientException;
import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Modified;

/**
 * @author mark
 * @since 13.05.2019
 */
@Component(name = "MinioClientImpl", configurationPid = MinioClient.CONFIGURATION_PID, configurationPolicy = ConfigurationPolicy.REQUIRE)
public class MinioClientImpl implements MinioClient {

    private static final String URL_PATTERN = "%s://%s:%s%s";
    private static final String AUTH_PATTERN = "AWS %s:%s";

    private volatile MinioConfig config;
    private volatile String name;

    private @interface MinioConfig {
        String name();

        String protocol() default "https";

        String hostname() default "localhost";

        int port() default 9000;

        String accessKey();

        String secretKey();

        int timeout() default 30;
    }

    /**
     * Called on component activation
     *
     * @param config the configuration
     * @throws ConfigurationException
     */
    @Activate
    public void activate(MinioConfig config) throws ConfigurationException {
        validateConfiguration(config);
        updateConfiguration(config);
    }

    /**
     * Called on component modification, when the configuration changes
     *
     * @param config the configuration
     * @throws ConfigurationException
     */
    @Modified
    public void modified(MinioConfig config) throws ConfigurationException {
        validateConfiguration(config);
        updateConfiguration(config);
    }

    /**
     * Called on component deactivation
     */
    @Deactivate
    public void deactivate() {
        // do nothing
    }

    /*
     * (non-Javadoc)
     * @see org.gecko.minio.MinioClient#getId()
     */
    @Override
    public String getId() {
        return name;
    }

    /*
     * (non-Javadoc)
     * @see org.gecko.minio.MinioClient#getFile(java.lang.String, java.lang.String)
     */
    @Override
    public InputStream getFile(String bucketPath, String contentType) throws MinioClientException {
        HttpURLConnection client = getConnection("GET", contentType, bucketPath);
        try {
            client.connect();

            if (client.getResponseCode() == 200) {
                return client.getInputStream();
            } else {
                InputStream response = client.getErrorStream();
                BufferedInputStream bis = new BufferedInputStream(response);
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                for (byte[] b = new byte[512]; 0 < bis.read(b); baos.write(b)) {
                    ;
                }
                String errorMessage = new String(baos.toByteArray(), StandardCharsets.UTF_8);
                throw new MinioClientException(String.format("[%s][%s] Response error: %s", name, client.getResponseCode(), errorMessage));
            }
        } catch (IOException e) {
            throw new MinioClientException(String.format("[%s] Error connecting client", name), e);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.gecko.minio.MinioClient#getFile(java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public InputStream getFile(String bucket, String file, String contentType) throws MinioClientException {
        return getFile("/" + bucket + "/" + file, contentType);
    }

    /*
     * (non-Javadoc)
     * @see org.gecko.minio.MinioClient#uploadFile(java.lang.String, java.lang.String, java.io.InputStream)
     */
    @Override
    public void uploadFile(String bucketPath, String contentType, InputStream input) throws MinioClientException {
        HttpURLConnection client = getConnection("PUT", contentType, bucketPath);
        client.setDoOutput(true);
        client.setChunkedStreamingMode(1024);
        try (DataOutputStream requestStream = new DataOutputStream(client.getOutputStream());
                BufferedInputStream bis = new BufferedInputStream(input);) {
            byte[] inputBuffer = new byte[1024];
            while (bis.read(inputBuffer) != -1) {
                requestStream.write(inputBuffer);
            }
            requestStream.flush();
            if (client.getResponseCode() != 200) {
                InputStream errorResponse = client.getErrorStream();
                BufferedInputStream errorBis = new BufferedInputStream(errorResponse);
                ByteArrayOutputStream errorBaos = new ByteArrayOutputStream();
                byte[] errorBuffer = new byte[512];
                while (0 < errorBis.read(errorBuffer)) {
                    errorBaos.write(errorBuffer);
                }
                String errorMessage = new String(errorBaos.toByteArray(), StandardCharsets.UTF_8);
                throw new MinioClientException(String.format("[%s][%s] Response error: %s", name, client.getResponseCode(), errorMessage));
            }
        } catch (IOException e) {
            throw new MinioClientException(String.format("[%s] Error connecting client", name), e);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.gecko.minio.MinioClient#uploadFile(java.lang.String, java.lang.String, java.lang.String, java.io.InputStream)
     */
    @Override
    public void uploadFile(String bucket, String file, String contentType, InputStream input) throws MinioClientException {
        uploadFile("/" + bucket + "/" + file, contentType, input);
    }

    /*
     * (non-Javadoc)
     * @see org.gecko.minio.MinioClient#deleteFile(java.lang.String)
     */
    @Override
    public void deleteFile(String bucketPath) throws MinioClientException {
        HttpURLConnection client = getConnection("DELETE", "application/octet-stream", bucketPath);
        try {
            client.connect();
            if (client.getResponseCode() != 204) {
                InputStream errorResponse = client.getErrorStream();
                BufferedInputStream errorBis = new BufferedInputStream(errorResponse);
                ByteArrayOutputStream errorBaos = new ByteArrayOutputStream();
                byte[] errorBuffer = new byte[512];
                while (0 < errorBis.read(errorBuffer)) {
                    errorBaos.write(errorBuffer);
                }
                String errorMessage = new String(errorBaos.toByteArray(), StandardCharsets.UTF_8);
                throw new MinioClientException(String.format("[%s][%s] Response error: %s", name, client.getResponseCode(), errorMessage));
            }
        } catch (IOException e) {
            throw new MinioClientException(String.format("[%s] Error connecting client", name), e);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.gecko.minio.MinioClient#deleteFile(java.lang.String, java.lang.String)
     */
    @Override
    public void deleteFile(String bucket, String file) throws MinioClientException {
        deleteFile("/" + bucket + "/" + file);
    }

    /**
     * Validates the configuration for missing properties
     *
     * @param config the configuration to validate
     * @throws ConfigurationException
     */
    private void validateConfiguration(MinioConfig config) throws ConfigurationException {
        if (config.name() == null) {
            throw new ConfigurationException("name", "A driver name must be provided");
        }
        if (config.accessKey() == null) {
            throw new ConfigurationException("accessKey", "An access key must be provided");
        }
        if (config.secretKey() == null) {
            throw new ConfigurationException("secretKey", "A secret key must be provided");
        }
    }

    /**
     * Updates the configuration on changed
     *
     * @param config the current new configuration
     */
    private void updateConfiguration(MinioConfig config) {
        if ( !sameConfigs(this.config, config)) {
            this.config = config;
            name = config.name();
        }
    }

    /**
     * Checks, if the configuration has changed
     *
     * @param oldConfig the old configuration
     * @param newConfig the current new configuration
     * @return <code>true</code>, if the configurations are same
     */
    private boolean sameConfigs(MinioConfig oldConfig, MinioConfig newConfig) {
        if (oldConfig == null && newConfig != null) {
            return false;
        }
        return oldConfig.accessKey().equals(newConfig.accessKey()) &&
                oldConfig.secretKey().equals(newConfig.secretKey()) &&
                oldConfig.name().equals(newConfig.name()) &&
                oldConfig.hostname().equals(newConfig.hostname()) &&
                oldConfig.protocol().equals(newConfig.protocol()) &&
                oldConfig.port() == newConfig.port();

    }

    /**
     * Creates a {@link HttpURLConnection} for the corresponding method and parameters
     *
     * @param method the HTTP method
     * @param contentType the content type
     * @param filePath the bucket file path
     * @return the {@link HttpURLConnection}
     * @throws MinioClientException
     */
    private HttpURLConnection getConnection(String method, String contentType, String filePath) throws MinioClientException {
        if (method == null || contentType == null || filePath == null) {
            throw new IllegalArgumentException(String.format("[%s] Error preparing a connection, because all parameters must not be null", name));
        }
        String dateString = MinioHelper.REQUEST_DATE_FORMATTER.format(new Date());
        try {
            String theUrl = String.format(URL_PATTERN, config.protocol(), config.hostname(), config.port(), filePath);
            URL url = new URL(theUrl);
            HttpURLConnection client = (HttpURLConnection) url.openConnection();
            client.setConnectTimeout(config.timeout());
            client.addRequestProperty("Host", config.hostname());
            client.addRequestProperty("Content-Type", contentType);
            client.addRequestProperty("Date", dateString);
            String hash = MinioHelper.generateAWSHash(method, contentType, dateString, filePath, config.secretKey());
            client.addRequestProperty("Authorization", String.format(AUTH_PATTERN, config.accessKey(), hash));
            return client;
        } catch (Exception e) {
            throw new MinioClientException("Error creating connection", e);
        }
    }

}
