/**
 * Copyright (c) 2012 - 2019 Data In Motion and others.
 * All rights reserved.
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * Data In Motion - initial API and implementation
 */
package org.gecko.minio.impl;

import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SignatureException;
import java.text.SimpleDateFormat;
import java.util.Base64;
import java.util.Locale;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

/**
 * Helper class for handling Minio communication
 * 
 * @author Mark Hoffmann
 * @since 13.05.2019
 */
public class MinioHelper {

    public static final SimpleDateFormat REQUEST_DATE_FORMATTER = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss Z", Locale.US);

    private static final String HMAC_SHA1_ALGORITHM = "HmacSHA1";
    private static final String HASH_PATTERN = "%s\n\n%s\n%s\n%s";

    /**
     * Calculates a Hmac SHA1 hash after RFC2104
     * 
     * @param content the content to be signed
     * @param secretKey the secret key to sign
     * @return the hash as byte array
     * @throws SignatureException
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     */
    public static byte[] generateHMACRaw(String content, String secretKey)
            throws SignatureException, NoSuchAlgorithmException, InvalidKeyException {
        if (content == null || secretKey == null) {
            throw new IllegalArgumentException("Both parameters must be set");
        }
        SecretKeySpec signingKey = new SecretKeySpec(secretKey.getBytes(), HMAC_SHA1_ALGORITHM);
        Mac mac = Mac.getInstance(HMAC_SHA1_ALGORITHM);
        mac.init(signingKey);
        return mac.doFinal(content.getBytes());
    }

    /**
     * Calculates a Hmac SHA1 hash after RFC2104, but Base64 encoded
     * 
     * @param content the content to be signed
     * @param secretKey the secret key to sign
     * @return the hash as byte array
     * @throws SignatureException
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     */
    public static byte[] generateHMACBase64(String content, String secretKey)
            throws SignatureException, NoSuchAlgorithmException, InvalidKeyException {
        byte[] unencoded = generateHMACRaw(content, secretKey);
        return Base64.getEncoder().encode(unencoded);
    }

    /**
     * Calculates a Hmac SHA1 hash after RFC2104, but Base64 encoded
     * 
     * @param content the content to be signed
     * @param secretKey the secret key to sign
     * @return the hash as String
     * @throws SignatureException
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     */
    public static String generateHMACBase64String(String content, String secretKey)
            throws SignatureException, NoSuchAlgorithmException, InvalidKeyException {
        byte[] unencoded = generateHMACRaw(content, secretKey);
        return Base64.getEncoder().encodeToString(unencoded);
    }

    /**
     * Generates the AWS hash out of the parameters
     * 
     * @param requestMethod the HTTP request method
     * @param contentType the content type
     * @param requestTimestamp the time in format 'EEE, dd MMM yyyy HH:mm:ss Z'
     * @param filePath the bucket file path
     * @return the AWS hash base 64 encoded
     * @throws NoSuchAlgorithmException
     * @throws SignatureException
     * @throws InvalidKeyException
     */
    public static String generateAWSHash(String requestMethod, String contentType, String requestTimestamp, String filePath, String secretKey)
            throws InvalidKeyException, SignatureException, NoSuchAlgorithmException {
        if (requestMethod == null || contentType == null || requestTimestamp == null || filePath == null || secretKey == null) {
            throw new IllegalArgumentException("Error generating AWS hash. All parameter must be set and must not be null");
        }
        String requestToSign = String.format(HASH_PATTERN, requestMethod, contentType, requestTimestamp, filePath);
        String hash = MinioHelper.generateHMACBase64String(requestToSign, secretKey);
        return hash;
    }


}
