/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.gecko.rsa.api.helper;

import java.util.Arrays;
import java.util.Collections;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.osgi.framework.Filter;
import org.osgi.service.remoteserviceadmin.EndpointDescription;

/**
 * End-point helper class
 */
public final class EndpointHelper {
	
	private static final Logger logger = Logger.getLogger(EndpointHelper.class.getName());

    private EndpointHelper() {
    }

    /**
     * Adds the given interfaces to the properties and creates the corresponding entry for that
     * @param properties the properties to add the interfaces into
     * @param interfaces the interfaces to be added
     */
    public static void addObjectClass(Map<String, Object> properties, Class<?>[] interfaces) {
    	if (properties == null) {
    		logger.warning("No properties are given to add some interfaces");
    		return;
    	}
    	if (interfaces == null || interfaces.length == 0) {
    		logger.warning("No interfaces are given to be added to the properties");
    		return;
    	}
        properties.put(org.osgi.framework.Constants.OBJECTCLASS, getClassNames(interfaces));
    }
    
    /**
     * Returns an array of class names for an array of classes
     * @param interfaces the array of classes / interfaces
     * @return the array of the class names
     */
    public static String[] getClassNames(Class<?>[] interfaces) {
    	return Arrays.asList(interfaces).stream().
    			map(i->i.getName()).
    			collect(Collectors.toList()).
    			toArray(new String[interfaces.length]);
    }
    
    /**
     * Returns all {@link Filter}'s that match the end-point properties
     * @param filters the filters to be checked
     * @param endpoint the end-point description with their properties 
     * @return a set of {@link Filter}'s that match or an empty {@link Set}
     */
    public static Set<Filter> getMatchingFilters(Set<Filter> filters, EndpointDescription endpoint) {
        if (endpoint == null || filters == null) {
        	logger.fine("No matching filter evaluated because endpoint or filters are null");
            return Collections.emptySet();
        }
        Dictionary<String, Object> epProperties = new Hashtable<String, Object>(endpoint.getProperties());
        return filters.stream().
        		filter(f->f.match(epProperties)).
        		collect(Collectors.toSet());
    }
}
