/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.gecko.rsa.api.helper;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.osgi.framework.Constants;
import org.osgi.framework.Filter;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.service.remoteserviceadmin.EndpointEventListener;
import org.osgi.service.remoteserviceadmin.RemoteConstants;

public final class FilterHelper {

	private static final Logger logger = Logger.getLogger(FilterHelper.class.getName());

	private static final String OBJECTCLASS_EXPRESSION = ".*\\(" + Constants.OBJECTCLASS + "=([^)]+)\\).*";
	private static final Pattern OBJECTCLASS_PATTERN = Pattern.compile(OBJECTCLASS_EXPRESSION);
	private static final Set<String> SYSTEM_PACKAGES;

	static {
		SYSTEM_PACKAGES = new HashSet<String>();
		SYSTEM_PACKAGES.add("org.osgi.service");
		SYSTEM_PACKAGES.add("org.apache.felix");
		SYSTEM_PACKAGES.add("org.ops4j.pax.logging");
		SYSTEM_PACKAGES.add("ch.ethz.iks.slp");
		SYSTEM_PACKAGES.add("org.ungoverned.osgi.service");
		SYSTEM_PACKAGES.add("org.springframework.osgi.context.event.OsgiBundleApplicationContextListener");
		SYSTEM_PACKAGES.add("java.net.ContentHandler");
	}

	private FilterHelper() {
		// prevent instantiation
	}

	/**
	 * Returns the object class from the given filter string
	 * @param filter the filter string to extract the object class from
	 * @return the object class or <code>null</code>
	 */
	public static String getObjectClass(String filter) {
		if (filter != null) {
			Matcher matcher = OBJECTCLASS_PATTERN.matcher(filter);
			if (matcher.matches() && matcher.groupCount() >= 1) {
				return matcher.group(1);
			}
		}
		return null;
	}

	/**
	 * Returns <code>true</code>, if the given string class name is part of the SYSTEM_PACKAGE list
	 * @param className the class name to check
	 * @return <code>true</code>, if the class name is <code>null</code> or part of the system package list
	 */
	public static boolean isClassExcluded(String className) {
		if (className == null) {
			return true;
		}
		return SYSTEM_PACKAGES.stream().
				filter((p)->className.startsWith(p)).
				findFirst().
				isPresent();
	}

	/**
	 * Returns an object class filter for a given object class and filter parameter
	 * @param objectClass the object class name
	 * @param filter an existing filter
	 * @return the filter string
	 */
	public static String getFullFilter(String objectClass, String filter) {
		if (objectClass == null) {
			return filter;
		}
		String nameFilter = String.format("(objectClass=%s)", objectClass); 
		return (filter == null) ? nameFilter : String.format("(&%s%s)", nameFilter, filter);
	}

	/**
	 * Extends a given filter with the framework UUID filter
	 * @param filter the given filter
	 * @param frameworkUUID the framework UUID
	 * @return the extended filter
	 */
	public static String extendFilter(String filter, String frameworkUUID) {
		return "(&" + filter + "(!(" + RemoteConstants.ENDPOINT_FRAMEWORK_UUID + "=" + frameworkUUID + ")))";
	}

	/**
	 * Extracts the scopes out of the service property from the service reference and converts it into {@link Filter}s
	 * @param eelReference the service reference of the end-point event listener
	 * @return a {@link Set} of filters or an empty {@link Set}
	 */
	public static Set<Filter> getFiltersFromEEL(ServiceReference<EndpointEventListener> eelReference) {
		List<String> scopes = FilterHelper.normalize(eelReference.getProperty(EndpointEventListener.ENDPOINT_LISTENER_SCOPE));
		return getFilterSet(scopes);
	}

	/**
	 * Creates a set of {@link Filter} out of the scope list, with filter as {@link String}
	 * @param scopes the filter as String
	 * @return the converted {@link Set} with {@link Filter} objects
	 */
	private static Set<Filter> getFilterSet(List<String> scopes) {
		Set<Filter> filters = new HashSet<Filter>();
		for (String scope : scopes) {
			try {
				filters.add(FrameworkUtil.createFilter(scope));
			} catch (InvalidSyntaxException e) {
				logger.log(Level.SEVERE, String.format("Invalid endpoint listener scope: %s", scope), e);
			}
		}
		return filters;
	}

	/**
	 * Normalizes the given input object to the corresponding rules. Currently comma separated strings, arrays and list are supported
	 * @param object the object to be normalized.
	 * @return a list of strings as result of the normalization
	 */
	public static List<String> normalize(Object object) {
		if (object instanceof String) {
			String string = (String)object;
			String[] values = string.split(",");
			return Arrays.asList(values).
					stream().
					map(String::trim).
					filter(s->!s.isEmpty()).
					collect(Collectors.toList());
		}
		if (object instanceof String[]) {
			return Arrays.asList((String[])object);
		}
		if (object instanceof Collection) {
			Collection<?> col = (Collection<?>)object;
			return col.stream().
					map(FilterHelper::logNonString).
					filter(e->e instanceof String).
					map(e->(String)e).
					collect(Collectors.toList());
		}
		return Collections.emptyList();
	}
	
	/**
	 * Logs, if the given element is no {@link String}
	 * @param object the object to check
	 * @return the object
	 */
	private static Object logNonString(Object object) {
		if (!(object instanceof String)) {
			logger.warning("Normalizing a collection contained non string element! this element was skipped");
		}
		return object;
	}

}
