/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.rsa.discovery.ma;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.aries.rsa.util.StringPlus;
import org.gecko.osgi.messaging.MessagingService;
import org.gecko.rsa.discovery.ma.publish.PublishingEndpointListener;
import org.gecko.rsa.discovery.ma.repository.MessageAdapterEndpointRepository;
import org.gecko.rsa.discovery.ma.subscribe.EndpointEventListenerManager;
import org.gecko.rsa.discovery.ma.subscribe.EndpointListenerTracker;
import org.gecko.rsa.discovery.ma.util.MADiscoveryConstants;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.framework.Filter;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceReference;
import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.cm.ManagedServiceFactory;
import org.osgi.service.remoteserviceadmin.EndpointDescription;
import org.osgi.service.remoteserviceadmin.EndpointEvent;
import org.osgi.service.remoteserviceadmin.EndpointEventListener;

/**
 * RSA discovery using Gecko MessageAdapter. This is inspired by the Zookeeper discovery.
 * @author Mark Hoffmann
 * @since 02.07.2018
 */
public class MessageAdapterDiscovery implements ManagedServiceFactory, MADiscoveryConstants {

	public static final String DISCOVERY_MA_ID = "org.gecko.rsa.discovery.ma";

	private static final Logger logger = Logger.getLogger(MessageAdapterDiscovery.class.getName());

	private Map<EndpointDescription, Bundle> endpointDescriptions =
			new ConcurrentHashMap<EndpointDescription, Bundle>();
	private Map<EndpointEventListener, Collection<String>> listenerToFilters =
			new ConcurrentHashMap<EndpointEventListener, Collection<String>>();
	private Map<String, Collection<EndpointEventListener>> filterToListeners =
			new ConcurrentHashMap<String, Collection<EndpointEventListener>>();
	private PublishingEndpointListener endpointListener;
	private BundleContext bctx;
	private MessageAdapterEndpointRepository repository;
	private EndpointEventListenerManager eelManager;
	private EndpointListenerTracker endpointListenerTracker;

	/**
	 * Creates a new instance.
	 */
	public MessageAdapterDiscovery(BundleContext bctx) {
		this.bctx = bctx;
	}

	/**
	 * Starts the message adapter discovery. It tracks local published from the topology manager as well as remote endpoints
	 * that are received from the message client
	 * @param messaging the messaging service
	 */
	public void start(MessagingService messaging) {
		// registering the handler for the messaging stuff
		repository = new MessageAdapterEndpointRepository(messaging, bctx.getProperty(Constants.FRAMEWORK_UUID));
		// register listener that listens for services to be exposed to other nodes
		endpointListener = new PublishingEndpointListener(repository);
		endpointListener.start(bctx);
		// create stuff for services that we received from other nodes
		eelManager = new EndpointEventListenerManager(repository);
		repository.setListener(eelManager);
		repository.initialize();
		endpointListenerTracker = new EndpointListenerTracker(bctx, eelManager);
		endpointListenerTracker.open();
	}

	/**
	 * Stops the messaging adapter discovery
	 */
	public void stop() {
		try {
			if (repository != null) {
				repository.close();
			}
			if (endpointListener != null) {
				endpointListener.stop();
			}
			if (endpointListenerTracker != null) {
				endpointListenerTracker.close();
			}
			if (eelManager != null) {
				eelManager.close();
			}
		} catch (IOException e) {
			logger.log(Level.SEVERE, "Error stopping MessageAdapter discovery, because of an IO error", e);
		} catch (Exception e) {
			logger.log(Level.SEVERE, "Error stopping MessageAdapter discovery", e);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see org.osgi.service.cm.ManagedServiceFactory#getName()
	 */
	@Override
	public String getName() {
		return "Gecko RSA MessageAdapter Discovery";
	}

	/* 
	 * (non-Javadoc)
	 * @see org.osgi.service.cm.ManagedServiceFactory#updated(java.lang.String, java.util.Dictionary)
	 */
	@Override
	public void updated(String pid, Dictionary<String, ?> properties) throws ConfigurationException {
	}

	/* 
	 * (non-Javadoc)
	 * @see org.osgi.service.cm.ManagedServiceFactory#deleted(java.lang.String)
	 */
	@Override
	public void deleted(String pid) {
	}

	/**
	 * Adds a new end-point listener
	 * @param reference the service reference
	 * @param endpointListener the end-point listener
	 */
	public void addListener(ServiceReference<EndpointEventListener> reference, EndpointEventListener endpointListener) {
		List<String> filters = StringPlus.normalize(reference.getProperty(EndpointEventListener.ENDPOINT_LISTENER_SCOPE));
		if (filters.isEmpty()) {
			return;
		}
		synchronized (listenerToFilters) {
			listenerToFilters.put(endpointListener, filters);
		}
		synchronized (filterToListeners) {
			for (String filter : filters) {
				Collection<EndpointEventListener> listeners = filterToListeners.get(filter);
				if (listeners == null) {
					listeners = new ArrayList<EndpointEventListener>();
					filterToListeners.put(filter, listeners);
				}
				listeners.add(endpointListener);
			}
		}
		triggerCallbacks(filters, endpointListener);
	}

	/**
	 * Removed an end-point listener
	 * @param endpointListener the listener to be removed
	 */
	public void removeListener(EndpointEventListener endpointListener) {
		synchronized (listenerToFilters) {
			Collection<String> filters = listenerToFilters.remove(endpointListener);
			if (filters == null) {
				return;
			}
			for (String filter : filters) {
				Collection<EndpointEventListener> listeners = filterToListeners.get(filter);
				if (listeners != null) {
					listeners.remove(endpointListener);
					if (listeners.isEmpty()) {
						filterToListeners.remove(filter);
					}
				}
			}
		}
	}

	/**
	 * Triggers end-point change event for the given event
	 * @param endpointListener the listener to notify
	 * @param filter the event filter
	 * @param event the event
	 */
	private void triggerCallbacks(EndpointEventListener endpointListener, String filter, EndpointEvent event) {
		if (!MessageAdapterDiscovery.matchFilter(filter, event.getEndpoint())) {
			return;
		}
		endpointListener.endpointChanged(event, filter);
	}

	/**
	 * Triggers end-point change events for a list of filters
	 * @param filters the list of filters
	 * @param endpointListener the listener to notify
	 */
	private void triggerCallbacks(Collection<String> filters, EndpointEventListener endpointListener) {
		for (String filter : filters) {
			synchronized (endpointDescriptions) {
				for (EndpointDescription endpoint : endpointDescriptions.keySet()) {
					EndpointEvent event = new EndpointEvent(EndpointEvent.ADDED, endpoint);
					triggerCallbacks(endpointListener, filter, event);
				}
			}
		}
	}

	/**
	 * Checks, if the given filter matches the end-point description
	 * @param filter the filter to check
	 * @param endpoint the end-point description
	 * @return <code>true</code>, if the filter matches
	 */
	private static boolean matchFilter(String filter, EndpointDescription endpoint) {
		if (filter == null) {
			return false;
		}
		try {
			Filter f = FrameworkUtil.createFilter(filter);
			Dictionary<String, Object> dict = new Hashtable<String, Object>(endpoint.getProperties());
			return f.match(dict);
		} catch (Exception e) {
			return false;
		}
	}

}
