/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.rsa.discovery.ma.converter;

import java.io.OutputStream;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.rsa.core.DeSerializationContext;
import org.gecko.rsa.core.EObjectDeAndSerializer;
import org.gecko.rsa.core.EObjectDeSerializationContext;
import org.gecko.rsa.core.SerializationContext;
import org.gecko.rsa.model.rsa.DocumentRoot;
import org.gecko.rsa.model.rsa.EndpointDescription;
import org.gecko.rsa.model.rsa.EndpointDescriptions;
import org.gecko.rsa.model.rsa.RSAFactory;
import org.gecko.rsa.model.rsa.RSAPackage;
import org.gecko.rsa.model.rsa.util.RSAResourceFactoryImpl;
import org.osgi.util.promise.Promise;

/**
 * Serializes and De-serializes the end-point descriptions
 * @author Mark Hoffmann
 * @since 24.07.2018
 */
public class EndPointDeSerializer extends EObjectDeAndSerializer<EndpointDescriptions, EndpointDescriptions> {
	
	/**
	 * Creates a new instance.
	 * @param dsContext
	 * @param sContext
	 */
	public EndPointDeSerializer(DeSerializationContext dsContext, SerializationContext sContext) {
		super(dsContext, sContext);
	}
	
	/**
	 * Creates a new instance with XML context.
	 * @param dsContext
	 * @param sContext
	 */
	public EndPointDeSerializer() {
		super(EObjectDeSerializationContext.getXMLDeSerializationContext(), EObjectDeSerializationContext.getXMLSerializationContext());
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.rsa.core.EObjectDeAndSerializer#doConfigureResourceSet(org.eclipse.emf.ecore.resource.ResourceSet)
	 */
	@Override
	protected void doConfigureResourceSet(ResourceSet resourceSet) {
		resourceSet.getPackageRegistry().put(RSAPackage.eNS_URI, RSAPackage.eINSTANCE);
		resourceSet.getResourceFactoryRegistry().getContentTypeToFactoryMap().put("application/xml", new RSAResourceFactoryImpl());
		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap().put("xml", new RSAResourceFactoryImpl());
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.rsa.core.EObjectDeAndSerializer#getSerializationContent(org.eclipse.emf.ecore.EObject)
	 */
	@Override
	protected EObject getSerializationContent(EndpointDescriptions serObject) {
		DocumentRoot root = RSAFactory.eINSTANCE.createDocumentRoot();
		root.setEndpointDescriptions(serObject);
		return root;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.rsa.core.EObjectDeAndSerializer#getDeSerializationContent(org.eclipse.emf.ecore.EObject)
	 */
	@Override
	protected EndpointDescriptions getDeSerializationContent(EObject content) {
		if (content == null || !(content instanceof DocumentRoot)) {
			throw new IllegalArgumentException("The expected parameter was null or not of type DocumentRoot");
		}
		DocumentRoot root = (DocumentRoot) content;
		return root.getEndpointDescriptions();
	}

	/**
	 * Easy serialize method, for just one end-point
	 * @param object the end-point description to be serialized
	 * @param output the output stream to serialize into
	 * @return promise for the output stream
	 */
	public Promise<OutputStream> serialize(EndpointDescription object, OutputStream output) {
		DocumentRoot root = RSAFactory.eINSTANCE.createDocumentRoot();
		EndpointDescriptions descriptions = RSAFactory.eINSTANCE.createEndpointDescriptions();
		root.setEndpointDescriptions(descriptions);
		descriptions.getEndpointDescription().add(object);
		return serialize(descriptions, output);
	}
	
	/**
	 * Easy serialize method, for just one end-point
	 * @param object the end-point description to be serialized
	 * @param output the output stream to serialize into
	 * @return promise for the output stream
	 */
	public Promise<OutputStream> serialize(EndpointDescription object) {
		DocumentRoot root = RSAFactory.eINSTANCE.createDocumentRoot();
		EndpointDescriptions descriptions = RSAFactory.eINSTANCE.createEndpointDescriptions();
		root.setEndpointDescriptions(descriptions);
		descriptions.getEndpointDescription().add(object);
		return serialize(descriptions);
	}

}
