/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.gecko.rsa.discovery.ma.publish;

import java.util.Dictionary;
import java.util.Hashtable;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.gecko.rsa.discovery.ma.MessageAdapterDiscovery;
import org.gecko.rsa.discovery.ma.repository.MessageAdapterEndpointRepository;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.remoteserviceadmin.EndpointDescription;
import org.osgi.service.remoteserviceadmin.EndpointEvent;
import org.osgi.service.remoteserviceadmin.EndpointEventListener;
import org.osgi.service.remoteserviceadmin.RemoteConstants;

/**
 * Listens for local EndpointEvents using new style listeners and publishes changes to 
 * the MessageAdapterEndpointRepository
 */
public class PublishingEndpointListener implements EndpointEventListener {

    private static final Logger logger = Logger.getLogger(PublishingEndpointListener.class.getName());

    private ServiceRegistration<?> listenerReg;
    private MessageAdapterEndpointRepository repository;

    public PublishingEndpointListener(MessageAdapterEndpointRepository repository) {
        this.repository = repository;
    }
    
    /**
     * Starts the endpoint listener for publishing endpoints
     * @param bctx the bundle context
     */
    public void start(BundleContext bctx) {
        Dictionary<String, String> props = new Hashtable<String, String>();
        String uuid = bctx.getProperty(Constants.FRAMEWORK_UUID);
        props.put(EndpointEventListener.ENDPOINT_LISTENER_SCOPE, 
                  String.format("(&(%s=*)(%s=%s))", Constants.OBJECTCLASS, 
                                RemoteConstants.ENDPOINT_FRAMEWORK_UUID, uuid));
        props.put(MessageAdapterDiscovery.DISCOVERY_MA_ID, "true");
        listenerReg = bctx.registerService(EndpointEventListener.class, this, props);
    }
    
    /**
     * Stops the endpoint listener for publishing endpoint descriptions
     */
    public void stop() {
        if (listenerReg != null) {
            listenerReg.unregister();
            listenerReg = null;
        }
    }

    /* 
     * (non-Javadoc)
     * @see org.osgi.service.remoteserviceadmin.EndpointEventListener#endpointChanged(org.osgi.service.remoteserviceadmin.EndpointEvent, java.lang.String)
     */
    @Override
    public void endpointChanged(EndpointEvent event, String filter) {
        EndpointDescription endpoint = event.getEndpoint();
        switch (event.getType()) {
        case EndpointEvent.ADDED:
            endpointAdded(endpoint, filter);
            break;
        case EndpointEvent.REMOVED:
            endpointRemoved(endpoint, filter);
            break;
        case EndpointEvent.MODIFIED:
            endpointModified(endpoint, filter);
            break;
        }
    }
    
    /**
     * Called if an end-point was modified
     * @param endpoint the end-point that was modified
     * @param filter the service filter
     */
    private void endpointModified(EndpointDescription endpoint, String filter) {
        try {
            repository.modify(endpoint);
        } catch (Exception e) {
            logger.log(Level.SEVERE, String.format("Error modifying endpoint data in zookeeper for endpoint %s", endpoint.getId() ), e);
        }
    }

    /**
     * Called, if an end-point was added
     * @param endpoint the end-point that was added
     * @param matchedFilter the filter
     */
    private void endpointAdded(EndpointDescription endpoint, String matchedFilter) {
        try {
            repository.add(endpoint);
        } catch (Exception ex) {
            logger.log(Level.SEVERE, "Exception while processing the addition of an endpoint.", ex);
        }
    }

    /**
     * Called, when an end-point was removed
     * @param endpoint the end-point that was removed
     * @param matchedFilter the filter
     */
    private void endpointRemoved(EndpointDescription endpoint, String matchedFilter) {
        try {
            repository.remove(endpoint);
        } catch (Exception ex) {
            logger.log(Level.SEVERE, "Exception while processing the removal of an endpoint", ex);
        }
    }

}
