/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.rsa.discovery.ma.repository;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.gecko.osgi.messaging.Message;
import org.gecko.osgi.messaging.MessagingService;
import org.gecko.rsa.discovery.ma.converter.EndPointDeSerializer;
import org.gecko.rsa.discovery.ma.converter.EndpointDescriptionConverter;
import org.gecko.rsa.discovery.ma.util.MADiscoveryConstants;
import org.gecko.rsa.model.rsa.EndpointDescriptions;
import org.gecko.rsa.model.rsa.Property;
import org.gecko.rsa.model.rsa.RSAFactory;
import org.osgi.service.remoteserviceadmin.EndpointDescription;
import org.osgi.service.remoteserviceadmin.EndpointEvent;
import org.osgi.service.remoteserviceadmin.EndpointEventListener;
import org.osgi.util.promise.Promise;
import org.osgi.util.pushstream.PushStream;

/**
 * End-point repository that handles out-going and in-coming messages. These are coming from events or turned into {@link EndpointEvent}
 * @author Mark Hoffmann
 * @since 06.07.2018
 */
public class MessageAdapterEndpointRepository implements Closeable {

	private static final Logger logger = Logger.getLogger(MessageAdapterEndpointRepository.class.getName());

	private final EndpointDescriptionConverter converter = new EndpointDescriptionConverter();
	private final EndPointDeSerializer deSerializer = new EndPointDeSerializer();
	private final MessagingService messaging;
	private final String frameworkUUID;
	private final Map<String, org.gecko.rsa.model.rsa.EndpointDescription> nodes = new ConcurrentHashMap<String, org.gecko.rsa.model.rsa.EndpointDescription>();
	private EndpointEventListener listener;

	private PushStream<Message> getSubscribe;
	private PushStream<Message> addSubscribe;
	private PushStream<Message> removeSubscribe;
	private PushStream<Message> modifySubscribe;
	private PushStream<Message> announceSubscribe;


	public MessageAdapterEndpointRepository(MessagingService messaging, String frameworkId) {
		this.messaging = messaging;
		this.frameworkUUID = frameworkId;
	}

	public MessageAdapterEndpointRepository(MessagingService messaging, EndpointEventListener listener, String frameworkId) {
		this.messaging = messaging;
		this.frameworkUUID = frameworkId;
		this.listener = listener;
	}

	/**
	 * Sets an addition {@link EndpointEventListener}, that is for the subscribing side, to inform the topology manager
	 * @param listener the {@link EndpointEventListener} to set
	 */
	public void setListener(EndpointEventListener listener) {
		this.listener = listener;
	}

	/**
	 * Initializes the repoository
	 */
	public void initialize() {
		try {
			addSubscribe = messaging.subscribe(MADiscoveryConstants.TOPIC_ADD_SERVICE);
			addSubscribe.forEach(this::notifyAdd);
		} catch (Exception e) {
			logger.log(Level.SEVERE, String.format("Error subscribing for topic ", MADiscoveryConstants.TOPIC_ADD_SERVICE), e);
		}
		try {
			removeSubscribe = messaging.subscribe(MADiscoveryConstants.TOPIC_REMOVE_SERVICE);
			removeSubscribe.forEach(this::notifyRemove);
		} catch (Exception e) {
			logger.log(Level.SEVERE, String.format("Error subscribing for topic ", MADiscoveryConstants.TOPIC_REMOVE_SERVICE), e);
		}
		try {
			modifySubscribe = messaging.subscribe(MADiscoveryConstants.TOPIC_MODIFY_SERVICE);
			modifySubscribe.forEach(this::notifyModify);
		} catch (Exception e) {
			logger.log(Level.SEVERE, String.format("Error subscribing for topic ", MADiscoveryConstants.TOPIC_MODIFY_SERVICE), e);
		}
		try {
			announceSubscribe = messaging.subscribe(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, frameworkUUID));
			announceSubscribe.forEach(this::notifyModify);
		} catch (Exception e) {
			logger.log(Level.SEVERE, String.format("Error subscribing for topic ", MADiscoveryConstants.TOPIC_MODIFY_SERVICE), e);
		}
		try {
			getSubscribe = messaging.subscribe(MADiscoveryConstants.TOPIC_GET_DESCRIPTION);
			getSubscribe.forEach(this::publishAllDescriptions);
		} catch (Exception e) {
			logger.log(Level.SEVERE, String.format("Error subscribing for topic ", MADiscoveryConstants.TOPIC_GET_DESCRIPTION), e);
		}
	}

	/**
	 * Adds an end-point description, that has to be exported
	 * @param endpoint the end-point to be added
	 * @throws InterruptedException
	 */
	public void add(EndpointDescription endpoint) throws InterruptedException  {
		org.gecko.rsa.model.rsa.EndpointDescription ed = converter.caseOSGiEndpointDescription(endpoint);
		String key = getKey(ed);
		if (key != null) {
			nodes.put(key, ed);
			publishEndpointDescription(ed, MADiscoveryConstants.TOPIC_ADD_SERVICE);
			logger.fine(String.format("Exporting endpoint via message adapter. Endpoint: %s, Key: %s", endpoint, key));
		} else {
			logger.severe(String.format("Exporting endpoint failed because of missing endpoint.id. Endpoint: %s, Key: %s" + ed, endpoint, key));
		}
	}

	/**
	 * Modifies an end-point description, that has to be exported
	 * @param endpoint the end-point to be modified
	 * @throws InterruptedException
	 */
	public void modify(EndpointDescription endpoint) throws InterruptedException {
		org.gecko.rsa.model.rsa.EndpointDescription ed = converter.caseOSGiEndpointDescription(endpoint);
		String key = getKey(ed);
		if (key != null) {
			nodes.put(key, ed);
			publishEndpointDescription(ed, MADiscoveryConstants.TOPIC_MODIFY_SERVICE);
			logger.info(String.format("Exporting changed endpoint via message adapter. Endpoint: %s, Key: %s", endpoint, key));
		} else {
			logger.severe(String.format("Exporting changed endpoint failed because of missing endpoint.id. Endpoint: %s, Key: %s" + ed, endpoint, key));
		}
	}

	/**
	 * Removed an end-point description, that has to be exported
	 * @param endpoint the end-point to be removed
	 * @throws InterruptedException
	 */
	public void remove(EndpointDescription endpoint) throws InterruptedException {
		org.gecko.rsa.model.rsa.EndpointDescription ed = converter.caseOSGiEndpointDescription(endpoint);
		String key = getKey(ed);
		if (key != null) {
			org.gecko.rsa.model.rsa.EndpointDescription remove = nodes.remove(getKey(ed));
			publishEndpointDescription(remove, MADiscoveryConstants.TOPIC_REMOVE_SERVICE);
			logger.info(String.format("Export remove endpoint via message adapter. Endpoint: %s, Path: %s removed: " + remove, endpoint, ""));
		} else {
			logger.severe(String.format("Export remove endpoint failed because of missing endpoint.id. Endpoint: %s, Key: %s" + ed, endpoint, key));
		}
	}

	/**
	 * Returns all end-point descriptions or an empty collection
	 * @return all end-point descriptions
	 */
	public Collection<org.gecko.rsa.model.rsa.EndpointDescription> getAll() {
		return nodes.values();
	}

	/* 
	 * (non-Javadoc)
	 * @see java.io.Closeable#close()
	 */
	@Override
	public void close() throws IOException {
		addSubscribe.close();
		removeSubscribe.close();
		modifySubscribe.close();
		announceSubscribe.close();
		getSubscribe.close();
		nodes.clear();
	}

	/**
	 * Removes nulls and empty strings from the given string array.
	 * @param strings an array of strings
	 * @return a new array containing the non-null and non-empty
	 *         elements of the original array in the same order
	 */
	public static List<String> removeEmpty(List<String> strings) {
		List<String> result = new ArrayList<String>();
		if (strings == null) {
			return result;
		}
		for (String s : strings) {
			if (s != null && !s.isEmpty()) {
				result.add(s);
			}
		}
		return result;
	}

	/**
	 * Returns the key, which is the end-point id
	 * @param endpointDescription the endpoint description
	 * @return the endpoint id property or null
	 */
	private String getKey(org.gecko.rsa.model.rsa.EndpointDescription endpointDescription) {
		Optional<Property> endpointId = endpointDescription.getProperty().stream().filter(p->p.getName().equals("endpoint.id")).findFirst();
		return endpointId.map(p->p.getValue()).orElse(null);
	}

	/**
	 * Publishes a end-point description holder
	 * @param descriptions the description holder
	 * @param topic the topic to publish
	 */
	private void publishEndpointDescription(org.gecko.rsa.model.rsa.EndpointDescriptions descriptions, String topic) {
		try {
			Promise<OutputStream> payload = deSerializer.serialize(descriptions);
			ByteArrayOutputStream baos = (ByteArrayOutputStream) payload.getValue();
			messaging.publish(topic, ByteBuffer.wrap(baos.toByteArray()));
		} catch (Exception e) {
			logger.log(Level.SEVERE, String.format("Error publishing payload for endpoint to topic %s", topic), e);
		}
	}

	/**
	 * Publishes a single end-point description
	 * @param description the end-point description
	 * @param topic the topic to publish
	 */
	private void publishEndpointDescription(org.gecko.rsa.model.rsa.EndpointDescription description, String topic) {
		EndpointDescriptions descriptions = RSAFactory.eINSTANCE.createEndpointDescriptions();
		descriptions.getEndpointDescription().add(description);
		publishEndpointDescription(descriptions, topic);
	}

	/**
	 * Publishes all descriptions
	 * @param message the message with payload
	 */
	private void publishAllDescriptions(Message message) {
		String topicPart = new String(message.payload().array());
		if (frameworkUUID == null || frameworkUUID.equalsIgnoreCase(topicPart)) {
			logger.fine("Received a get all request for this OSGi framework (Requesting ourself). It can be ignored.");
			return;
		}
		String topic = String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, topicPart);
		Collection<org.gecko.rsa.model.rsa.EndpointDescription> allDescritpions = getAll();
		if (!allDescritpions.isEmpty()) {
			EndpointDescriptions descriptions = RSAFactory.eINSTANCE.createEndpointDescriptions();
			descriptions.getEndpointDescription().addAll(allDescritpions);
			publishEndpointDescription(descriptions, topic);
		}
	}

	/**
	 * Received a add message and notifies all listeners about this end-point change
	 * @param message the received message
	 */
	private void notifyAdd(Message message) {
		if (listener == null) {
			logger.warning("Cannot consume add message with end-point description because no listener is configured");
			return;
		}
		ByteArrayInputStream bais = new ByteArrayInputStream(message.payload().array());
		Promise<EndpointDescriptions> endpointDescriptionsProm = deSerializer.deserialize(bais);
		EndpointDescriptions endpointDescriptions;
		try {
			endpointDescriptions = endpointDescriptionsProm.getValue();
			for (org.gecko.rsa.model.rsa.EndpointDescription ed : endpointDescriptions.getEndpointDescription()) {
				String key = getKey(ed);
				if (nodes.containsKey(key)) {
					logger.info("Detected and new incoming endpoint, that already exists. Ignore it");
					continue;
				} 
				EndpointDescription endpointDescription = converter.doSwitch(ed);
				nodes.put(key, ed);
				EndpointEvent event = new EndpointEvent(EndpointEvent.ADDED, endpointDescription);
				listener.endpointChanged(event, null);
				logger.info("Detected and incoming endpoint andsent 'ADD' event");
			}
		} catch (InvocationTargetException | InterruptedException e) {
			logger.log(Level.SEVERE, "Detected an error for an incoming endpoint. Cannot send ADD event", e);
		}
	}

	/**
	 * Received a modification message and notifies all listeners about this end-point change.
	 * If there is no end-point description, it will be created and an add event will be sent. 
	 * Otherwise a modified event will be sent
	 * @param message the received message
	 */
	private void notifyModify(Message message) {
		if (listener == null) {
			logger.warning("Cannot consume modify message with end-point description because no listener is configured");
			return;
		}
		ByteArrayInputStream bais = new ByteArrayInputStream(message.payload().array());
		Promise<EndpointDescriptions> endpointDescriptionsProm = deSerializer.deserialize(bais);
		EndpointDescriptions endpointDescriptions;
		try {
			endpointDescriptions = endpointDescriptionsProm.getValue();
			for (org.gecko.rsa.model.rsa.EndpointDescription ed : endpointDescriptions.getEndpointDescription()) {
				String key = getKey(ed);
				EndpointDescription endpointDescription = converter.doSwitch(ed);
				org.gecko.rsa.model.rsa.EndpointDescription edModify = nodes.put(key, ed);
				int eventType = EndpointEvent.ADDED;
				if (edModify != null) {
					eventType = EndpointEvent.MODIFIED;
				}
				EndpointEvent event = new EndpointEvent(eventType, endpointDescription);
				listener.endpointChanged(event, null);
				logger.info(String.format("Detected and incoming endpoint sent '%s' event", (eventType == EndpointEvent.ADDED ? "ADD" : "MODIFY")));
			}
		} catch (InvocationTargetException | InterruptedException e) {
			logger.log(Level.SEVERE, "Detected an error for a modified endpoint. Cannot send MODIFY or ADD event", e);
		}
	}

	/**
	 * Received a remove message and notifies all listeners about this end-point change.
	 * @param message the received message
	 */
	private void notifyRemove(Message message) {
		if (listener == null) {
			logger.warning("Cannot consume remove message with end-point description because no listener is configured");
			return;
		}
		ByteArrayInputStream bais = new ByteArrayInputStream(message.payload().array());
		Promise<EndpointDescriptions> endpointDescriptionsProm = deSerializer.deserialize(bais);
		EndpointDescriptions endpointDescriptions;
		try {
			endpointDescriptions = endpointDescriptionsProm.getValue();
			for (org.gecko.rsa.model.rsa.EndpointDescription ed : endpointDescriptions.getEndpointDescription()) {
				String key = getKey(ed);
				org.gecko.rsa.model.rsa.EndpointDescription edRemove = nodes.remove(key);
				if (edRemove != null) {
					EndpointDescription endpointDescription = converter.doSwitch(edRemove);
					EndpointEvent event = new EndpointEvent(EndpointEvent.REMOVED, endpointDescription);
					listener.endpointChanged(event, null);
					logger.info("Detected and incoming endpoint sent REMOVE event");
				} else {
					logger.info("No endpoint was found, that could be removed");
				}
			}
		} catch (InvocationTargetException | InterruptedException e) {
			logger.log(Level.SEVERE, "Detected an error for a remove endpoint. Cannot send REOMVE event", e);
		}
	}

	/**
	 * Triggers a get all endpoints message
	 */
	public void initializeTopologyManager() {
		try {
			messaging.publish(MADiscoveryConstants.TOPIC_GET_DESCRIPTION, ByteBuffer.wrap(frameworkUUID.getBytes()));
		} catch (Exception e) {
			logger.log(Level.SEVERE, String.format("Error sending intialize get all request for topic ", MADiscoveryConstants.TOPIC_GET_DESCRIPTION), e);
		}
	}

}
