/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.gecko.rsa.discovery.ma.subscribe;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;

import org.apache.aries.rsa.util.StringPlus;
import org.gecko.rsa.discovery.ma.MessageAdapterDiscovery;
import org.gecko.rsa.discovery.ma.converter.EndpointDescriptionConverter;
import org.gecko.rsa.discovery.ma.repository.MessageAdapterEndpointRepository;
import org.gecko.rsa.model.rsa.EndpointDescription;
import org.osgi.framework.ServiceReference;
import org.osgi.service.remoteserviceadmin.EndpointEvent;
import org.osgi.service.remoteserviceadmin.EndpointEventListener;

/**
 * Manages the EndpointEventListeners and the scopes they are interested in.
 * Establishes a listener with the repository to be called back on all changes in the repo.
 * Events from repository are then forwarded to all interested EndpointEventListeners.
 */
@SuppressWarnings({"rawtypes"})
public class EndpointEventListenerManager implements EndpointEventListener {
	
    private static final Logger logger = Logger.getLogger(EndpointEventListenerManager.class.getName());
    private final EndpointDescriptionConverter converter = new EndpointDescriptionConverter();
    private final MessageAdapterEndpointRepository repository;
    private final Map<ServiceReference, Interest> epWhiteboards = new ConcurrentHashMap<ServiceReference, Interest>();

    protected static class Interest {
        List<String> scopes;
        EndpointEventListener epListener;
    }

    public EndpointEventListenerManager(MessageAdapterEndpointRepository repository) {
        this.repository = repository;
    }

    /**
     * Adds or modifies an end-point event listener
     * @param eelRef the service reference 
     * @param epListener the listener instance
     */
    public void modifyEPListener(ServiceReference<EndpointEventListener> eelRef, EndpointEventListener epListener, int eventType) {
        if (isOurOwnEndpointEventListener(eelRef)) {
            logger.fine("Skipping our own EndpointEventListener");
            return;
        }
        List<String> scopes = getScopes(eelRef);
        
        // get or create interest for given scope and add listener to it
        synchronized (epWhiteboards) {
        	Interest interest = epWhiteboards.get(eelRef);
        	if (interest == null) {
        		// create interest, add listener and start monitor
        		interest = new Interest();
        		epWhiteboards.put(eelRef, interest);
        		repository.initializeTopologyManager();
        	} 
    		interest.epListener = epListener;
    		interest.scopes = scopes;
    		sendExistingEndpoints(scopes, epListener, eventType);
		}
    }

    /**
     * Removed an end-point event listener
     * @param epListenerRef the service reference
     */
    public synchronized void removeEPListener(ServiceReference<EndpointEventListener> epListenerRef) {
	    logger.finest(String.format("Removing EndpointEventListener whiteboard: %s", epListenerRef));
	    synchronized (epWhiteboards) {
	    	epWhiteboards.remove(epListenerRef);
		}
	}

	/**
	 * Closes the manager
	 */
	public synchronized void close() {
		synchronized (epWhiteboards) {
			epWhiteboards.clear();
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see org.osgi.service.remoteserviceadmin.EndpointEventListener#endpointChanged(org.osgi.service.remoteserviceadmin.EndpointEvent, java.lang.String)
	 */
	@Override
	public void endpointChanged(EndpointEvent event, String filter) {
		synchronized (epWhiteboards) {
			for (Interest interest : epWhiteboards.values()) {
				notifyListener(event, interest.scopes, interest.epListener);
			}
		}
	}

	private void sendExistingEndpoints(List<String> scopes, EndpointEventListener endpointListener, int eventType) {
        for (EndpointDescription endpoint : repository.getAll()) {
        	org.osgi.service.remoteserviceadmin.EndpointDescription ed = converter.doSwitch(endpoint);
            EndpointEvent event = new EndpointEvent(eventType, ed);
            notifyListener(event, scopes, endpointListener);
        }
    }

    /**
     * Notifies the given end-point listener for end-point changed, if the scope of the {@link org.osgi.service.remoteserviceadmin.EndpointDescription}
     * matches one of the scopes in the given list
     * @param event the {@link EndpointEvent}
     * @param scopes the list of scopes
     * @param endpointListener the end-point listener to be notified
     */
    private void notifyListener(EndpointEvent event, List<String> scopes, EndpointEventListener endpointListener) {
        org.osgi.service.remoteserviceadmin.EndpointDescription endpoint = event.getEndpoint();
        String currentScope = getFirstMatch(scopes, endpoint);
        if (currentScope == null) {
        	logger.fine(String.format("Current scope is null for endpoint event %s", event));
            return;
        }
        logger.fine(String.format("Calling endpoint changed on end-point listener for scope %s, endpoint %s ", currentScope, endpoint));
        endpointListener.endpointChanged(event, currentScope);
    }
    
    /**
     * Returns the first scope in the given list, that matches to the {@link org.osgi.service.remoteserviceadmin.EndpointDescription}
     * @param scopes list of scopes
     * @param endpoint the end-point description
     * @return the first scope or <code>null</code>
     */
    private String getFirstMatch(List<String> scopes, org.osgi.service.remoteserviceadmin.EndpointDescription endpoint) {
	    for (String scope : scopes) {
	        if (endpoint.matches(scope)) {
	            return scope;
	        }
	    }
	    return null;
	}

	/**
     * Returns a list of {@link EndpointEventListener#ENDPOINT_LISTENER_SCOPE} or <code>null</code>
     * @param serviceReference the service reference to get the scope from
     * @return a list of {@link EndpointEventListener#ENDPOINT_LISTENER_SCOPE} or <code>null</code>
     */
    protected List<String> getScopes(ServiceReference<?> serviceReference) {
        return StringPlus.normalize(serviceReference.getProperty(EndpointEventListener.ENDPOINT_LISTENER_SCOPE));
    }

	/**
	 * Returns <code>true</code>, if we are the given end-point listener.
	 * @param endpointEventListener the service reference for the end-point listener to be checked
	 * @return
	 */
	private static boolean isOurOwnEndpointEventListener(ServiceReference<EndpointEventListener> endpointEventListener) {
	    return Boolean.parseBoolean(String.valueOf(
	    		endpointEventListener.getProperty(MessageAdapterDiscovery.DISCOVERY_MA_ID)));
	}

}
