/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.gecko.rsa.discovery.ma.subscribe;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.remoteserviceadmin.EndpointEvent;
import org.osgi.service.remoteserviceadmin.EndpointEventListener;
import org.osgi.util.tracker.ServiceTracker;

/**
 * Tracks EndpointEventListeners. Delegates to EndpointEventListenerManager to handle them
 * @author Mark Hoffmann
 * @since 06.07.2018
 */
public class EndpointListenerTracker extends ServiceTracker<EndpointEventListener, EndpointEventListener> {
	
    private final EndpointEventListenerManager eelManager;

    public EndpointListenerTracker(BundleContext bctx, EndpointEventListenerManager eelManager) {
        super(bctx, EndpointEventListener.class, null);
        this.eelManager = eelManager;
    }
    
    /* 
     * (non-Javadoc)
     * @see org.osgi.util.tracker.ServiceTracker#addingService(org.osgi.framework.ServiceReference)
     */
    @Override
    public EndpointEventListener addingService(ServiceReference<EndpointEventListener> sref) {
    	EndpointEventListener epListener = super.addingService(sref);
        eelManager.modifyEPListener(sref, epListener, EndpointEvent.ADDED);
        return epListener;
    }

    /* 
     * (non-Javadoc)
     * @see org.osgi.util.tracker.ServiceTracker#modifiedService(org.osgi.framework.ServiceReference, java.lang.Object)
     */
    @Override
    public void modifiedService(ServiceReference<EndpointEventListener> sref, EndpointEventListener epListener) {
        // called when an EndpointListener updates its service properties,
        // e.g. when its interest scope is expanded/reduced
        eelManager.modifyEPListener(sref, epListener, EndpointEvent.MODIFIED);
    }

    /* 
     * (non-Javadoc)
     * @see org.osgi.util.tracker.ServiceTracker#removedService(org.osgi.framework.ServiceReference, java.lang.Object)
     */
    @Override
    public void removedService(ServiceReference<EndpointEventListener> sref, EndpointEventListener epListener) {
        eelManager.removeEPListener(sref);
        super.removedService(sref, epListener);
    }

}
