/**
 * 
 */
package org.gecko.rsa.core.tests.ma;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.anyString;
import static org.mockito.Matchers.argThat;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.lang.annotation.Annotation;
import java.nio.ByteBuffer;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import org.eclipse.emf.ecore.util.FeatureMapUtil;
import org.gecko.core.tests.AbstractOSGiTest;
import org.gecko.core.tests.ServiceChecker;
import org.gecko.osgi.messaging.Message;
import org.gecko.osgi.messaging.MessagingContext;
import org.gecko.osgi.messaging.MessagingService;
import org.gecko.rsa.core.EndPointDeSerializer;
import org.gecko.rsa.discovery.ma.handler.MADiscoveryConfig;
import org.gecko.rsa.discovery.ma.handler.MADiscoveryConstants;
import org.gecko.rsa.discovery.ma.handler.MessageAdapterDiscoverHandler;
import org.gecko.rsa.model.rsa.Array;
import org.gecko.rsa.model.rsa.EndpointDescription;
import org.gecko.rsa.model.rsa.Property;
import org.gecko.rsa.model.rsa.RSAFactory;
import org.gecko.rsa.model.rsa.Value;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentMatcher;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.Constants;
import org.osgi.framework.FrameworkUtil;
import org.osgi.service.remoteserviceadmin.EndpointEvent;
import org.osgi.service.remoteserviceadmin.EndpointEventListener;
import org.osgi.util.promise.Promise;
import org.osgi.util.pushstream.PushStreamProvider;
import org.osgi.util.pushstream.SimplePushEventSource;

/**
 * @author mark
 *
 */
@RunWith(MockitoJUnitRunner.class)
public class MessagingHandlerTest extends AbstractOSGiTest {
	
	@Mock
	MessagingService messaging;
	private MessageAdapterDiscoverHandler madh;
	private EndPointDeSerializer deSerializer = new EndPointDeSerializer();
	
	public MessagingHandlerTest() {
		super(FrameworkUtil.getBundle(MessagingHandlerTest.class).getBundleContext());
	}

	/* (non-Javadoc)
	 * @see org.gecko.core.tests.AbstractOSGiTest#doBefore()
	 */
	@Override
	public void doBefore() {
	}

	/* (non-Javadoc)
	 * @see org.gecko.core.tests.AbstractOSGiTest#doAfter()
	 */
	@Override
	public void doAfter() {
		if (madh != null) {
			madh.stop();
		}
	}
	
	/**
	 * Test a successful export of a discovery of an internal service
	 * @throws Exception
	 */
	@Test
	public void testExportAddDescriptionsSuccess() throws Exception {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		Map<String, Object> configProperties = new HashMap<String, Object>();
		configProperties.put("name", "test");
		MADiscoveryConfig config = getConfiguration(configProperties);
		
		/*
		 * Mocking subscribe methods used in handler, just to avoid some exception in the test
		 */
		PushStreamProvider psp = new PushStreamProvider();
		SimplePushEventSource<Message> pesAdd = psp.createSimpleEventSource(Message.class);
		SimplePushEventSource<Message> pes = psp.createSimpleEventSource(Message.class);
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_ADD_SERVICE)).thenReturn(psp.createStream(pesAdd));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_GET_DESCRIPTION)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_MODIFY_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_REMOVE_SERVICE)).thenReturn(psp.createStream(pes));
		String fwid = getBundleContext().getProperty(Constants.FRAMEWORK_UUID);
		when(messaging.subscribe(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, fwid))).thenReturn(psp.createStream(pes));
		
		madh = new MessageAdapterDiscoverHandler(messaging, getBundleContext(), config);
		madh.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointEventListener eventListener = endpointListenerTracker.getTrackedService();
		assertNotNull(eventListener);
		
		// Send event description add from topology manager side
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test1");
		properties.put(Constants.OBJECTCLASS, new String[] {Object.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e = new EndpointEvent(EndpointEvent.ADDED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		eventListener.endpointChanged(e, null);
		
		Thread.sleep(1000l);
		
		// Verify correct call sttistics of message adapters
		verify(messaging, times(1)).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_MODIFY_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_REMOVE_SERVICE), any(ByteBuffer.class));
		verify(messaging).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), argThat(new ArgumentMatcher<ByteBuffer>() {

			@Override
			public boolean matches(Object argument) {
				assertTrue(argument instanceof ByteBuffer);
				byte[] content = ((ByteBuffer)argument).array();
				assertNotNull(content);
				String contentString = new String(content);
				assertTrue(contentString.contains("test1"));
				return true;
			}
			
		}));;
		
		madh.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	/**
	 * Test a modification of a description, that is new at all. So an add event will be sent 
	 * @throws Exception
	 */
	@Test
	public void testExportModifyAddDescriptionsSuccess() throws Exception {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		Map<String, Object> configProperties = new HashMap<String, Object>();
		configProperties.put("name", "test");
		MADiscoveryConfig config = getConfiguration(configProperties);
		
		/*
		 * Mocking subscribe methods used in handler, just to avoid some exception in the test
		 */
		PushStreamProvider psp = new PushStreamProvider();
		SimplePushEventSource<Message> pes = psp.createSimpleEventSource(Message.class);
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_ADD_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_GET_DESCRIPTION)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_MODIFY_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_REMOVE_SERVICE)).thenReturn(psp.createStream(pes));
		String fwid = getBundleContext().getProperty(Constants.FRAMEWORK_UUID);
		when(messaging.subscribe(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, fwid))).thenReturn(psp.createStream(pes));
		
		madh = new MessageAdapterDiscoverHandler(messaging, getBundleContext(), config);
		madh.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointEventListener eventListener = endpointListenerTracker.getTrackedService();
		assertNotNull(eventListener);
		
		// send the modification event
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test1");
		properties.put(Constants.OBJECTCLASS, new String[] {Object.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e = new EndpointEvent(EndpointEvent.MODIFIED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		eventListener.endpointChanged(e, null);
		
		Thread.sleep(1000l);
		
		// We expect an add event instead of a modify, because this service is new to our discovery
		verify(messaging, times(1)).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_MODIFY_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_REMOVE_SERVICE), any(ByteBuffer.class));
		verify(messaging).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), argThat(new ArgumentMatcher<ByteBuffer>() {
			
			@Override
			public boolean matches(Object argument) {
				assertTrue(argument instanceof ByteBuffer);
				byte[] content = ((ByteBuffer)argument).array();
				assertNotNull(content);
				String contentString = new String(content);
				assertTrue(contentString.contains("test1"));
				return true;
			}
			
		}));;
		
		madh.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	/**
	 * Tests the export of a description modification, with the corresponding add before
	 * @throws Exception
	 */
	@Test
	public void testExportModifyDescriptionsSuccess() throws Exception {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		Map<String, Object> configProperties = new HashMap<String, Object>();
		configProperties.put("name", "test");
		MADiscoveryConfig config = getConfiguration(configProperties);
		
		/*
		 * Mocking subscribe methods used in handler, just to avoid some exception in the test
		 */
		PushStreamProvider psp = new PushStreamProvider();
		SimplePushEventSource<Message> pes = psp.createSimpleEventSource(Message.class);
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_ADD_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_GET_DESCRIPTION)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_MODIFY_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_REMOVE_SERVICE)).thenReturn(psp.createStream(pes));
		String fwid = getBundleContext().getProperty(Constants.FRAMEWORK_UUID);
		when(messaging.subscribe(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, fwid))).thenReturn(psp.createStream(pes));
		
		madh = new MessageAdapterDiscoverHandler(messaging, getBundleContext(), config);
		madh.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointEventListener eventListener = endpointListenerTracker.getTrackedService();
		assertNotNull(eventListener);
		
		// notify about a new description
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test1");
		properties.put(Constants.OBJECTCLASS, new String[] {Object.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e01 = new EndpointEvent(EndpointEvent.ADDED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		eventListener.endpointChanged(e01, null);
		
		// notify about a modifcation description
		properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test1");
		properties.put(Constants.OBJECTCLASS, new String[] {String.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e02 = new EndpointEvent(EndpointEvent.MODIFIED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		eventListener.endpointChanged(e02, null);
		
		Thread.sleep(1000l);
		
		// check the call statistics, one add and one modify. And check the real changes
		verify(messaging, times(1)).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), any(ByteBuffer.class));
		verify(messaging, times(1)).publish(eq(MADiscoveryConstants.TOPIC_MODIFY_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_REMOVE_SERVICE), any(ByteBuffer.class));
		verify(messaging).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), argThat(new ArgumentMatcher<ByteBuffer>() {
			
			@Override
			public boolean matches(Object argument) {
				assertTrue(argument instanceof ByteBuffer);
				byte[] content = ((ByteBuffer)argument).array();
				assertNotNull(content);
				String contentString = new String(content);
				assertTrue(contentString.contains("test1"));
				assertTrue(contentString.contains("java.lang.Object"));
				return true;
			}
			
		}));
		verify(messaging).publish(eq(MADiscoveryConstants.TOPIC_MODIFY_SERVICE), argThat(new ArgumentMatcher<ByteBuffer>() {
			
			@Override
			public boolean matches(Object argument) {
				assertTrue(argument instanceof ByteBuffer);
				byte[] content = ((ByteBuffer)argument).array();
				assertNotNull(content);
				String contentString = new String(content);
				assertTrue(contentString.contains("test1"));
				assertTrue(contentString.contains("java.lang.String"));
				return true;
			}
			
		}));
		
		madh.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	/**
	 * Tests the removal of a description with an additional add
	 * @throws Exception
	 */
	@Test
	public void testExportRemoveDescriptionsSuccess() throws Exception {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		Map<String, Object> configProperties = new HashMap<String, Object>();
		configProperties.put("name", "test");
		MADiscoveryConfig config = getConfiguration(configProperties);
		
		/*
		 * Mocking subscribe methods used in handler, just to avoid some exception in the test
		 */
		PushStreamProvider psp = new PushStreamProvider();
		SimplePushEventSource<Message> pes = psp.createSimpleEventSource(Message.class);
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_ADD_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_GET_DESCRIPTION)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_MODIFY_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_REMOVE_SERVICE)).thenReturn(psp.createStream(pes));
		String fwid = getBundleContext().getProperty(Constants.FRAMEWORK_UUID);
		when(messaging.subscribe(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, fwid))).thenReturn(psp.createStream(pes));
		
		madh = new MessageAdapterDiscoverHandler(messaging, getBundleContext(), config);
		madh.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointEventListener eventListener = endpointListenerTracker.getTrackedService();
		assertNotNull(eventListener);
		
		// send add description
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test1");
		properties.put(Constants.OBJECTCLASS, new String[] {Object.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e01 = new EndpointEvent(EndpointEvent.ADDED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		eventListener.endpointChanged(e01, null);
		
		// send remove description
		properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test1");
		properties.put(Constants.OBJECTCLASS, new String[] {Object.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e02 = new EndpointEvent(EndpointEvent.REMOVED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		eventListener.endpointChanged(e02, null);
		
		Thread.sleep(1000l);
		
		// verify call statistic, we expect one add and one remove
		verify(messaging, times(1)).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), any(ByteBuffer.class));
		verify(messaging, times(1)).publish(eq(MADiscoveryConstants.TOPIC_REMOVE_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_MODIFY_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), any(ByteBuffer.class));
		verify(messaging).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), argThat(new ArgumentMatcher<ByteBuffer>() {
			
			@Override
			public boolean matches(Object argument) {
				assertTrue(argument instanceof ByteBuffer);
				byte[] content = ((ByteBuffer)argument).array();
				assertNotNull(content);
				String contentString = new String(content);
				assertTrue(contentString.contains("test1"));
				return true;
			}
			
		}));
		verify(messaging).publish(eq(MADiscoveryConstants.TOPIC_REMOVE_SERVICE), argThat(new ArgumentMatcher<ByteBuffer>() {
			
			@Override
			public boolean matches(Object argument) {
				assertTrue(argument instanceof ByteBuffer);
				byte[] content = ((ByteBuffer)argument).array();
				assertNotNull(content);
				String contentString = new String(content);
				assertTrue(contentString.contains("test1"));
				return true;
			}
			
		}));
		
		madh.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	/**
	 * Test the export of a removal without an add before 
	 * @throws Exception
	 */
	@Test
	public void testExportRemoveNoDescriptionsSuccess() throws Exception {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		Map<String, Object> configProperties = new HashMap<String, Object>();
		configProperties.put("name", "test");
		MADiscoveryConfig config = getConfiguration(configProperties);
		
		/*
		 * Mocking subscribe methods used in handler, just to avoid some exception in the test
		 */
		PushStreamProvider psp = new PushStreamProvider();
		SimplePushEventSource<Message> pes = psp.createSimpleEventSource(Message.class);
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_ADD_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_GET_DESCRIPTION)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_MODIFY_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_REMOVE_SERVICE)).thenReturn(psp.createStream(pes));
		String fwid = getBundleContext().getProperty(Constants.FRAMEWORK_UUID);
		when(messaging.subscribe(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, fwid))).thenReturn(psp.createStream(pes));
		
		madh = new MessageAdapterDiscoverHandler(messaging, getBundleContext(), config);
		madh.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointEventListener eventListener = endpointListenerTracker.getTrackedService();
		assertNotNull(eventListener);
		
		// send the remove description
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test1");
		properties.put(Constants.OBJECTCLASS, new String[] {Object.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e02 = new EndpointEvent(EndpointEvent.REMOVED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		eventListener.endpointChanged(e02, null);
		
		Thread.sleep(1000l);
		
		// verify that no publish method was called
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_REMOVE_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_MODIFY_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), any(ByteBuffer.class));
		
		madh.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	/**
	 * Test importing new description via message adapter
	 * @throws Exception
	 */
	@Test
	public void testImportAddDescriptionsSuccess() throws Exception {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("name", "test");
		MADiscoveryConfig config = getConfiguration(properties);
		
		// register correct pushstream for topic in mock
		PushStreamProvider psp = new PushStreamProvider();
		SimplePushEventSource<Message> pes = psp.createSimpleEventSource(Message.class);
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_ADD_SERVICE)).thenReturn(psp.createStream(pes));
		
		madh = new MessageAdapterDiscoverHandler(messaging, getBundleContext(), config);
		madh.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		// create import description and message
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("endpoint.id");
		p0101.setValue("1");
		d01.getProperty().add(p0101);
		Property p0102 = RSAFactory.eINSTANCE.createProperty();
		p0102.setName(Constants.OBJECTCLASS);
		Array a0102 = RSAFactory.eINSTANCE.createArray();
		Value v010201 = RSAFactory.eINSTANCE.createValue();
		a0102.getValue().add(v010201);
		FeatureMapUtil.addText(v010201.getMixed(), Object.class.getName());
		p0102.setArray(a0102);
		d01.getProperty().add(p0102);
		Property p0103 = RSAFactory.eINSTANCE.createProperty();
		p0103.setName("service.imported.configs");
		p0103.setValue("test.cnf");
		d01.getProperty().add(p0103);
		Promise<OutputStream> serialize = deSerializer.serialize(d01);
		Message message = serialize.map(s->(ByteArrayOutputStream)s).map(this::getMessage).getValue();
		assertNotNull(message);
		
		// create topology manager mock and register it
		AtomicBoolean added = new AtomicBoolean(false);
		CountDownLatch addLatch = new CountDownLatch(1);
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				if (EndpointEvent.ADDED == event.getType()) {
					added.set(true);
					addLatch.countDown();
				}
				
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(EndpointEventListener.ENDPOINT_LISTENER_SCOPE, "(service.imported.configs=test.cnf)");
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		// send importing message over registered channel
		pes.publish(message);
		
		assertTrue(addLatch.await(3, TimeUnit.SECONDS));
		assertTrue(added.get());
		
		// verify call statistic, that no publish was called
		verify(messaging, never()).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_MODIFY_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_REMOVE_SERVICE), any(ByteBuffer.class));
		
		madh.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	/**
	 * Test an incoming changed description. 
	 * @throws Exception
	 */
	@Test
	public void testImportModifyDescriptionsSuccess() throws Exception {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("name", "test");
		MADiscoveryConfig config = getConfiguration(properties);
		
		// register correct pushstream for topic in mock
		PushStreamProvider psp = new PushStreamProvider();
		SimplePushEventSource<Message> pes = psp.createSimpleEventSource(Message.class);
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_MODIFY_SERVICE)).thenReturn(psp.createStream(pes));
		
		madh = new MessageAdapterDiscoverHandler(messaging, getBundleContext(), config);
		madh.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		// create import description and message
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("endpoint.id");
		p0101.setValue("1");
		d01.getProperty().add(p0101);
		Property p0102 = RSAFactory.eINSTANCE.createProperty();
		p0102.setName(Constants.OBJECTCLASS);
		Array a0102 = RSAFactory.eINSTANCE.createArray();
		Value v010201 = RSAFactory.eINSTANCE.createValue();
		a0102.getValue().add(v010201);
		FeatureMapUtil.addText(v010201.getMixed(), Object.class.getName());
		p0102.setArray(a0102);
		d01.getProperty().add(p0102);
		Property p0103 = RSAFactory.eINSTANCE.createProperty();
		p0103.setName("service.imported.configs");
		p0103.setValue("test.cnf");
		d01.getProperty().add(p0103);
		Promise<OutputStream> serialize = deSerializer.serialize(d01);
		Message message = serialize.map(s->(ByteArrayOutputStream)s).map(this::getMessage).getValue();
		assertNotNull(message);
		
		AtomicBoolean modified = new AtomicBoolean(false);
		CountDownLatch modifyLatch = new CountDownLatch(1);
		AtomicBoolean added = new AtomicBoolean(false);
		CountDownLatch addLatch = new CountDownLatch(1);
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				if (EndpointEvent.ADDED == event.getType()) {
					added.set(true);
					addLatch.countDown();
				}
				if (EndpointEvent.MODIFIED == event.getType()) {
					modified.set(true);
					modifyLatch.countDown();
				}
				
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(EndpointEventListener.ENDPOINT_LISTENER_SCOPE, "(service.imported.configs=test.cnf)");
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		// send the message over the configured channels -> ends in an add
		pes.publish(message);
		
		assertTrue(addLatch.await(3, TimeUnit.SECONDS));
		assertTrue(added.get());
		assertFalse(modified.get());
		
		// send the message over the configured channels -> ends in an modify
		pes.publish(message);
		
		assertTrue(modifyLatch.await(3, TimeUnit.SECONDS));
		assertTrue(added.get());
		assertTrue(modified.get());
		
		// verify call statistic, that no publish was called
		verify(messaging, never()).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_MODIFY_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_REMOVE_SERVICE), any(ByteBuffer.class));
		
		madh.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testRemoveImportDescriptionsSuccess() throws Exception {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("name", "test");
		MADiscoveryConfig config = getConfiguration(properties);
		
		// register correct pushstream for topic in mock
		PushStreamProvider psp = new PushStreamProvider();
		SimplePushEventSource<Message> pesAdd = psp.createSimpleEventSource(Message.class);
		SimplePushEventSource<Message> pesRemove = psp.createSimpleEventSource(Message.class);
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_ADD_SERVICE)).thenReturn(psp.createStream(pesAdd));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_REMOVE_SERVICE)).thenReturn(psp.createStream(pesRemove));
		
		
		madh = new MessageAdapterDiscoverHandler(messaging, getBundleContext(), config);
		madh.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("endpoint.id");
		p0101.setValue("1");
		d01.getProperty().add(p0101);
		Property p0102 = RSAFactory.eINSTANCE.createProperty();
		p0102.setName(Constants.OBJECTCLASS);
		Array a0102 = RSAFactory.eINSTANCE.createArray();
		Value v010201 = RSAFactory.eINSTANCE.createValue();
		a0102.getValue().add(v010201);
		FeatureMapUtil.addText(v010201.getMixed(), Object.class.getName());
		p0102.setArray(a0102);
		d01.getProperty().add(p0102);
		Property p0103 = RSAFactory.eINSTANCE.createProperty();
		p0103.setName("service.imported.configs");
		p0103.setValue("test.cnf");
		d01.getProperty().add(p0103);
		Promise<OutputStream> serialize = deSerializer.serialize(d01);
		Message message = serialize.map(s->(ByteArrayOutputStream)s).map(this::getMessage).getValue();
		assertNotNull(message);
		
		AtomicBoolean added = new AtomicBoolean(false);
		CountDownLatch addLatch = new CountDownLatch(1);
		AtomicBoolean removed = new AtomicBoolean(false);
		CountDownLatch removeLatch = new CountDownLatch(1);
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				if (EndpointEvent.ADDED == event.getType()) {
					added.set(true);
					addLatch.countDown();
				}
				if (EndpointEvent.REMOVED == event.getType()) {
					removed.set(true);
					removeLatch.countDown();
				}
				
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(EndpointEventListener.ENDPOINT_LISTENER_SCOPE, "(service.imported.configs=test.cnf)");
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		pesAdd.publish(message);
		
		assertTrue(addLatch.await(3, TimeUnit.SECONDS));
		assertTrue(added.get());
		assertFalse(removed.get());
		
		pesRemove.publish(message);
		
		assertTrue(removeLatch.await(3, TimeUnit.SECONDS));
		assertTrue(added.get());
		assertTrue(removed.get());
		
		// verify call statistic, that no publish was called
		verify(messaging, never()).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_MODIFY_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_REMOVE_SERVICE), any(ByteBuffer.class));
		
		madh.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testGetAllRequestImporting() throws Exception {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("name", "test");
		MADiscoveryConfig config = getConfiguration(properties);
		
		PushStreamProvider psp = new PushStreamProvider();
		SimplePushEventSource<Message> pes = psp.createSimpleEventSource(Message.class);
		SimplePushEventSource<Message> pesGet = psp.createSimpleEventSource(Message.class);
		SimplePushEventSource<Message> pesAdd = psp.createSimpleEventSource(Message.class);
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_GET_DESCRIPTION)).thenReturn(psp.createStream(pesGet));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_ADD_SERVICE)).thenReturn(psp.createStream(pesAdd));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_MODIFY_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_REMOVE_SERVICE)).thenReturn(psp.createStream(pes));
		String fwid = getBundleContext().getProperty(Constants.FRAMEWORK_UUID);
		when(messaging.subscribe(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, fwid))).thenReturn(psp.createStream(pes));
		
		madh = new MessageAdapterDiscoverHandler(messaging, getBundleContext(), config);
		madh.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("endpoint.id");
		p0101.setValue("ep1");
		d01.getProperty().add(p0101);
		Property p0102 = RSAFactory.eINSTANCE.createProperty();
		p0102.setName(Constants.OBJECTCLASS);
		Array a0102 = RSAFactory.eINSTANCE.createArray();
		Value v010201 = RSAFactory.eINSTANCE.createValue();
		a0102.getValue().add(v010201);
		FeatureMapUtil.addText(v010201.getMixed(), Object.class.getName());
		p0102.setArray(a0102);
		d01.getProperty().add(p0102);
		Property p0103 = RSAFactory.eINSTANCE.createProperty();
		p0103.setName("service.imported.configs");
		p0103.setValue("test.cnf");
		d01.getProperty().add(p0103);
		Promise<OutputStream> serialize01 = deSerializer.serialize(d01);
		Message message01 = serialize01.map(s->(ByteArrayOutputStream)s).map(this::getMessage).getValue();
		assertNotNull(message01);
		
		EndpointDescription d02 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0201 = RSAFactory.eINSTANCE.createProperty();
		p0201.setName("endpoint.id");
		p0201.setValue("ep2");
		d02.getProperty().add(p0201);
		Property p0202 = RSAFactory.eINSTANCE.createProperty();
		p0202.setName(Constants.OBJECTCLASS);
		Array a0202 = RSAFactory.eINSTANCE.createArray();
		Value v020201 = RSAFactory.eINSTANCE.createValue();
		a0202.getValue().add(v020201);
		FeatureMapUtil.addText(v020201.getMixed(), Object.class.getName());
		p0202.setArray(a0202);
		d02.getProperty().add(p0202);
		Property p0203 = RSAFactory.eINSTANCE.createProperty();
		p0203.setName("service.imported.configs");
		p0203.setValue("test.cnf");
		d02.getProperty().add(p0203);
		Promise<OutputStream> serialize02 = deSerializer.serialize(d02);
		Message message02 = serialize02.map(s->(ByteArrayOutputStream)s).map(this::getMessage).getValue();
		assertNotNull(message02);
		
		AtomicInteger added = new AtomicInteger();
		CountDownLatch addLatch = new CountDownLatch(2);
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				if (EndpointEvent.ADDED == event.getType()) {
					added.incrementAndGet();
					addLatch.countDown();
				}
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(EndpointEventListener.ENDPOINT_LISTENER_SCOPE, "(service.imported.configs=test.cnf)");
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		pesAdd.publish(message01);
		pesAdd.publish(message02);
		
		assertTrue(addLatch.await(3, TimeUnit.SECONDS));
		assertEquals(2, added.get());
		
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		baos.write("TestFW".getBytes());
		baos.flush();
		Message getMessage = getMessage(baos);
		pesGet.publish(getMessage);
		
		Thread.sleep(1000l);
		
		// 5 subscriptions in MessageAdapterDiscoveryHandler
		verify(messaging, times(5)).subscribe(anyString());
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_MODIFY_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_REMOVE_SERVICE), any(ByteBuffer.class));
		verify(messaging, times(1)).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), any(ByteBuffer.class));
		verify(messaging).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), argThat(new ArgumentMatcher<ByteBuffer>() {

			@Override
			public boolean matches(Object argument) {
				assertTrue(argument instanceof ByteBuffer);
				byte[] content = ((ByteBuffer)argument).array();
				assertNotNull(content);
				String contentString = new String(content);
				assertFalse(contentString.contains("ep1"));
				assertFalse(contentString.contains("ep2"));
				return true;
			}
			
		}));
		
		madh.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testGetAllExportingImporting() throws Exception {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		Map<String, Object> configProperties = new HashMap<String, Object>();
		configProperties.put("name", "test");
		MADiscoveryConfig config = getConfiguration(configProperties);
		
		PushStreamProvider psp = new PushStreamProvider();
		SimplePushEventSource<Message> pes = psp.createSimpleEventSource(Message.class);
		SimplePushEventSource<Message> pesGet = psp.createSimpleEventSource(Message.class);
		SimplePushEventSource<Message> pesAdd = psp.createSimpleEventSource(Message.class);
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_GET_DESCRIPTION)).thenReturn(psp.createStream(pesGet));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_ADD_SERVICE)).thenReturn(psp.createStream(pesAdd));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_MODIFY_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_REMOVE_SERVICE)).thenReturn(psp.createStream(pes));
		String fwid = getBundleContext().getProperty(Constants.FRAMEWORK_UUID);
		when(messaging.subscribe(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, fwid))).thenReturn(psp.createStream(pes));
		
		madh = new MessageAdapterDiscoverHandler(messaging, getBundleContext(), config);
		madh.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("endpoint.id");
		p0101.setValue("ep1");
		d01.getProperty().add(p0101);
		Property p0102 = RSAFactory.eINSTANCE.createProperty();
		p0102.setName(Constants.OBJECTCLASS);
		Array a0102 = RSAFactory.eINSTANCE.createArray();
		Value v010201 = RSAFactory.eINSTANCE.createValue();
		a0102.getValue().add(v010201);
		FeatureMapUtil.addText(v010201.getMixed(), Object.class.getName());
		p0102.setArray(a0102);
		d01.getProperty().add(p0102);
		Property p0103 = RSAFactory.eINSTANCE.createProperty();
		p0103.setName("service.imported.configs");
		p0103.setValue("test.cnf");
		d01.getProperty().add(p0103);
		Promise<OutputStream> serialize01 = deSerializer.serialize(d01);
		Message message01 = serialize01.map(s->(ByteArrayOutputStream)s).map(this::getMessage).getValue();
		assertNotNull(message01);
		
		AtomicInteger added = new AtomicInteger();
		CountDownLatch addLatch = new CountDownLatch(1);
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				if (EndpointEvent.ADDED == event.getType()) {
					added.incrementAndGet();
					addLatch.countDown();
				}
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(EndpointEventListener.ENDPOINT_LISTENER_SCOPE, "(service.imported.configs=test.cnf)");
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointEventListener eventListener = endpointListenerTracker.getTrackedService();
		assertNotNull(eventListener);
		
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test1");
		properties.put(Constants.OBJECTCLASS, new String[] {Object.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e01 = new EndpointEvent(EndpointEvent.ADDED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		eventListener.endpointChanged(e01, null);
		
		Thread.sleep(1000l);
		
		pesAdd.publish(message01);
		
		assertTrue(addLatch.await(3, TimeUnit.SECONDS));
		assertEquals(1, added.get());
		
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		baos.write("TestFW".getBytes());
		baos.flush();
		Message getMessage = getMessage(baos);
		pesGet.publish(getMessage);
		
		Thread.sleep(1000l);
		
		// 5 subscriptions in MessageAdapterDiscoveryHandler
		verify(messaging, times(5)).subscribe(anyString());
		// 1 time for the sent add event -> test1
		verify(messaging, times(1)).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_MODIFY_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_REMOVE_SERVICE), any(ByteBuffer.class));
		verify(messaging, times(1)).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), any(ByteBuffer.class));
		verify(messaging).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), argThat(new ArgumentMatcher<ByteBuffer>() {
			
			@Override
			public boolean matches(Object argument) {
				assertTrue(argument instanceof ByteBuffer);
				byte[] content = ((ByteBuffer)argument).array();
				assertNotNull(content);
				String contentString = new String(content);
				assertTrue(contentString.contains("test1"));
				assertFalse(contentString.contains("ep2"));
				return true;
			}
			
		}));
		
		madh.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testGetAllRequestExporting() throws Exception {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		Map<String, Object> configProperties = new HashMap<String, Object>();
		configProperties.put("name", "test");
		MADiscoveryConfig config = getConfiguration(configProperties);
		
		PushStreamProvider psp = new PushStreamProvider();
		SimplePushEventSource<Message> pes = psp.createSimpleEventSource(Message.class);
		SimplePushEventSource<Message> pesGet = psp.createSimpleEventSource(Message.class);
		SimplePushEventSource<Message> pesAdd = psp.createSimpleEventSource(Message.class);
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_GET_DESCRIPTION)).thenReturn(psp.createStream(pesGet));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_ADD_SERVICE)).thenReturn(psp.createStream(pesAdd));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_MODIFY_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_REMOVE_SERVICE)).thenReturn(psp.createStream(pes));
		String fwid = getBundleContext().getProperty(Constants.FRAMEWORK_UUID);
		when(messaging.subscribe(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, fwid))).thenReturn(psp.createStream(pes));
		
		madh = new MessageAdapterDiscoverHandler(messaging, getBundleContext(), config);
		madh.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		EndpointEventListener eventListener = endpointListenerTracker.getTrackedService();
		assertNotNull(eventListener);
		
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test1");
		properties.put(Constants.OBJECTCLASS, new String[] {Object.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e01 = new EndpointEvent(EndpointEvent.ADDED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		eventListener.endpointChanged(e01, null);
		
		properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test2");
		properties.put(Constants.OBJECTCLASS, new String[] {String.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e02 = new EndpointEvent(EndpointEvent.ADDED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		eventListener.endpointChanged(e02, null);
		
		Thread.sleep(1000l);
		
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		baos.write("TestFW".getBytes());
		baos.flush();
		Message getMessage = getMessage(baos);
		pesGet.publish(getMessage);
		
		Thread.sleep(1000l);
		
		// 5 subscriptions in MessageAdapterDiscoveryHandler
		verify(messaging, times(5)).subscribe(anyString());
		verify(messaging, times(2)).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_MODIFY_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_REMOVE_SERVICE), any(ByteBuffer.class));
		verify(messaging, times(1)).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), any(ByteBuffer.class));
		verify(messaging).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), argThat(new ArgumentMatcher<ByteBuffer>() {
			
			@Override
			public boolean matches(Object argument) {
				assertTrue(argument instanceof ByteBuffer);
				byte[] content = ((ByteBuffer)argument).array();
				assertNotNull(content);
				String contentString = new String(content);
				assertTrue(contentString.contains("test1"));
				assertTrue(contentString.contains("test2"));
				return true;
			}
			
		}));
		
		madh.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	/**
	 * Test requesting all descriptions on startup
	 * @throws Exception
	 */
	@Test
	public void testRequestAll() throws Exception {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		Map<String, Object> configProperties = new HashMap<String, Object>();
		configProperties.put("name", "test");
		MADiscoveryConfig config = getConfiguration(configProperties);
		
		PushStreamProvider psp = new PushStreamProvider();
		SimplePushEventSource<Message> pes = psp.createSimpleEventSource(Message.class);
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_GET_DESCRIPTION)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_ADD_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_MODIFY_SERVICE)).thenReturn(psp.createStream(pes));
		when(messaging.subscribe(MADiscoveryConstants.TOPIC_REMOVE_SERVICE)).thenReturn(psp.createStream(pes));
		String fwid = getBundleContext().getProperty(Constants.FRAMEWORK_UUID);
		when(messaging.subscribe(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, fwid))).thenReturn(psp.createStream(pes));
		
		madh = new MessageAdapterDiscoverHandler(messaging, getBundleContext(), config);
		madh.start();
		
		Thread.sleep(1000l);
		
		// 5 subscriptions in MessageAdapterDiscoveryHandler
		verify(messaging, times(1)).publish(eq(MADiscoveryConstants.TOPIC_GET_DESCRIPTION), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_ADD_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_MODIFY_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(MADiscoveryConstants.TOPIC_REMOVE_SERVICE), any(ByteBuffer.class));
		verify(messaging, never()).publish(eq(String.format(MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE, "TestFW")), any(ByteBuffer.class));
		verify(messaging).publish(eq(MADiscoveryConstants.TOPIC_GET_DESCRIPTION), argThat(new ArgumentMatcher<ByteBuffer>() {
			
			@Override
			public boolean matches(Object argument) {
				assertTrue(argument instanceof ByteBuffer);
				byte[] content = ((ByteBuffer)argument).array();
				assertNotNull(content);
				String contentString = new String(content);
				assertEquals(fwid, contentString);
				return true;
			}
			
		}));
		
		madh.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	private MADiscoveryConfig getConfiguration(Map<String, Object> properties) {
		return new MADiscoveryConfig() {
			
			@Override
			public Class<? extends Annotation> annotationType() {
				return MADiscoveryConfig.class;
			}
			
			@Override
			public String removeTopic() {
				return (String) properties.getOrDefault("removeTopic", MADiscoveryConstants.TOPIC_REMOVE_SERVICE);
			}
			
			@Override
			public String name() {
				return (String) properties.get("name");
			}
			
			@Override
			public String modifyTopic() {
				return (String) properties.getOrDefault("modifyTopic", MADiscoveryConstants.TOPIC_MODIFY_SERVICE);
			}
			
			@Override
			public String initialGetResponse() {
				return (String) properties.getOrDefault("initialGetResponse", MADiscoveryConstants.TOPIC_GET_DESCRIPTION);
			}
			
			@Override
			public String importDescription() {
				return (String) properties.getOrDefault("importDescription", MADiscoveryConstants.TOPIC_ANNOUNCED_SERVICE);
			}
			
			@Override
			public String addTopic() {
				return (String) properties.getOrDefault("addTopic", MADiscoveryConstants.TOPIC_ADD_SERVICE);
			}
		}; 
	}
	
	private Message getMessage(ByteArrayOutputStream baos) {
		ByteBuffer buffer = ByteBuffer.wrap(baos.toByteArray());
		return new Message() {
			
			@Override
			public String topic() {
				return "testTopic";
			}
			
			@Override
			public ByteBuffer payload() {
				return buffer;
			}
			
			@Override
			public MessagingContext getContext() {
				return null;
			}
		};
	}

}
