package org.gecko.rsa.core.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import org.gecko.core.tests.AbstractOSGiTest;
import org.gecko.core.tests.ServiceChecker;
import org.gecko.rsa.api.ExportEndpointHandler;
import org.gecko.rsa.discovery.EndpointDiscovery;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.Constants;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.service.remoteserviceadmin.EndpointEvent;
import org.osgi.service.remoteserviceadmin.EndpointEventListener;

@RunWith(MockitoJUnitRunner.class)
public class ExportingDiscoveryIntegrationTest extends AbstractOSGiTest {
	
	private EndpointDiscovery discovery01;
	private EndpointDiscovery discovery02;

	public ExportingDiscoveryIntegrationTest() {
		super(FrameworkUtil.getBundle(ExportingDiscoveryIntegrationTest.class).getBundleContext());
	}

	@Override
	public void doBefore() {
	}
	
	@Override
	public void doAfter() {
		if (discovery01 != null && discovery01.isRunning()) {
			discovery01.stop();
		}
		if (discovery02 != null && discovery02.isRunning()) {
			discovery02.stop();
		}
	}
	
	@Test
	public void testEndpointListenerRegistration() {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		discovery01.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testExportDescriptionAdd() throws InterruptedException {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		AtomicBoolean added = new AtomicBoolean(false);
		AtomicBoolean modified = new AtomicBoolean(false);
		AtomicBoolean removed = new AtomicBoolean(false);
		CountDownLatch addLatch = new CountDownLatch(1);
		ExportEndpointHandler exportHandler = new ExportEndpointHandler() {
			
			@Override
			public void removeServiceDescription(org.osgi.service.remoteserviceadmin.EndpointDescription endpoint)
					throws Exception {
				removed.set(true);
			}
			
			@Override
			public void modifyServiceDescription(org.osgi.service.remoteserviceadmin.EndpointDescription endpoint)
					throws Exception {
				modified.set(true);
			}
			
			@Override
			public void exportServiceDescription(org.osgi.service.remoteserviceadmin.EndpointDescription endpoint)
					throws Exception {
				added.set(true);
				addLatch.countDown();
			}
		};
		discovery01.start();
		discovery01.getExportingEndpointListener().setExportHandler(exportHandler);
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointEventListener eventListener = endpointListenerTracker.getTrackedService();
		assertNotNull(eventListener);
		
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test");
		properties.put(Constants.OBJECTCLASS, new String[] {Object.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e = new EndpointEvent(EndpointEvent.ADDED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		eventListener.endpointChanged(e, null);
		
		assertTrue(addLatch.await(3, TimeUnit.SECONDS));
		assertTrue(added.get());
		assertFalse(modified.get());
		assertFalse(removed.get());
		
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testExportDescriptionModify() throws InterruptedException {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		AtomicBoolean added = new AtomicBoolean(false);
		AtomicBoolean modified = new AtomicBoolean(false);
		AtomicBoolean removed = new AtomicBoolean(false);
		CountDownLatch modifyLatch = new CountDownLatch(1);
		ExportEndpointHandler exportHandler = new ExportEndpointHandler() {
			
			@Override
			public void removeServiceDescription(org.osgi.service.remoteserviceadmin.EndpointDescription endpoint)
					throws Exception {
				removed.set(true);
			}
			
			@Override
			public void modifyServiceDescription(org.osgi.service.remoteserviceadmin.EndpointDescription endpoint)
					throws Exception {
				modified.set(true);
				modifyLatch.countDown();
			}
			
			@Override
			public void exportServiceDescription(org.osgi.service.remoteserviceadmin.EndpointDescription endpoint)
					throws Exception {
				added.set(true);
			}
		};
		discovery01.start();
		discovery01.getExportingEndpointListener().setExportHandler(exportHandler);
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointEventListener eventListener = endpointListenerTracker.getTrackedService();
		assertNotNull(eventListener);
		
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test");
		properties.put(Constants.OBJECTCLASS, new String[] {Object.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e = new EndpointEvent(EndpointEvent.MODIFIED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		eventListener.endpointChanged(e, null);
		
		assertTrue(modifyLatch.await(3, TimeUnit.SECONDS));
		assertTrue(modified.get());
		assertFalse(added.get());
		assertFalse(removed.get());
		
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testExportDescriptionRemove() throws InterruptedException {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		AtomicBoolean added = new AtomicBoolean(false);
		AtomicBoolean modified = new AtomicBoolean(false);
		AtomicBoolean removed = new AtomicBoolean(false);
		CountDownLatch removeLatch = new CountDownLatch(1);
		ExportEndpointHandler exportHandler = new ExportEndpointHandler() {
			
			@Override
			public void removeServiceDescription(org.osgi.service.remoteserviceadmin.EndpointDescription endpoint)
					throws Exception {
				removed.set(true);
				removeLatch.countDown();
			}
			
			@Override
			public void modifyServiceDescription(org.osgi.service.remoteserviceadmin.EndpointDescription endpoint)
					throws Exception {
				modified.set(true);
			}
			
			@Override
			public void exportServiceDescription(org.osgi.service.remoteserviceadmin.EndpointDescription endpoint)
					throws Exception {
				added.set(true);
			}
		};
		discovery01.start();
		discovery01.getExportingEndpointListener().setExportHandler(exportHandler);
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointEventListener eventListener = endpointListenerTracker.getTrackedService();
		assertNotNull(eventListener);
		
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test");
		properties.put(Constants.OBJECTCLASS, new String[] {Object.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e = new EndpointEvent(EndpointEvent.REMOVED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		eventListener.endpointChanged(e, null);
		
		assertTrue(removeLatch.await(3, TimeUnit.SECONDS));
		assertTrue(removed.get());
		assertFalse(added.get());
		assertFalse(modified.get());
		
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testExportDescriptionMultiDiscovery() throws InterruptedException, InvalidSyntaxException {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		discovery02 = new EndpointDiscovery(getBundleContext(), "Second");
		AtomicInteger added = new AtomicInteger();
		AtomicInteger modified = new AtomicInteger();
		AtomicInteger removed = new AtomicInteger();
		CountDownLatch latch = new CountDownLatch(3);
		ExportEndpointHandler exportHandler = new ExportEndpointHandler() {
			
			@Override
			public void removeServiceDescription(org.osgi.service.remoteserviceadmin.EndpointDescription endpoint)
					throws Exception {
				removed.incrementAndGet();
				latch.countDown();
			}
			
			@Override
			public void modifyServiceDescription(org.osgi.service.remoteserviceadmin.EndpointDescription endpoint)
					throws Exception {
				modified.incrementAndGet();
				latch.countDown();
			}
			
			@Override
			public void exportServiceDescription(org.osgi.service.remoteserviceadmin.EndpointDescription endpoint)
					throws Exception {
				added.incrementAndGet();
				latch.countDown();
			}
		};
		discovery01.start();
		discovery01.getExportingEndpointListener().setExportHandler(exportHandler);
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		discovery02.start();
		discovery02.getExportingEndpointListener().setExportHandler(exportHandler);
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointEventListener testDiscovery = getService(FrameworkUtil.createFilter("(Test=true)"), 3000);
		assertNotNull(testDiscovery);
		EndpointEventListener secondDiscovery = getService(FrameworkUtil.createFilter("(Second=true)"), 3000);
		assertNotNull(secondDiscovery);
		
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("endpoint.id", "test");
		properties.put(Constants.OBJECTCLASS, new String[] {Object.class.getName()});
		properties.put("service.imported.configs", "test.cnf");
		EndpointEvent e01 = new EndpointEvent(EndpointEvent.ADDED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		testDiscovery.endpointChanged(e01, null);
		EndpointEvent e02 = new EndpointEvent(EndpointEvent.REMOVED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		secondDiscovery.endpointChanged(e02, null);
		EndpointEvent e03 = new EndpointEvent(EndpointEvent.MODIFIED, new org.osgi.service.remoteserviceadmin.EndpointDescription(properties));
		testDiscovery.endpointChanged(e03, null);
		
		assertTrue(latch.await(3, TimeUnit.SECONDS));
		assertEquals(1, added.get());
		assertEquals(1, modified.get());
		assertEquals(1, removed.get());
		
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
		discovery02.stop();
		assertEquals(2, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
}
