package org.gecko.rsa.core.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.Dictionary;
import java.util.Hashtable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import org.eclipse.emf.ecore.util.FeatureMapUtil;
import org.gecko.core.tests.AbstractOSGiTest;
import org.gecko.core.tests.ServiceChecker;
import org.gecko.rsa.discovery.EndpointDiscovery;
import org.gecko.rsa.model.rsa.Array;
import org.gecko.rsa.model.rsa.EndpointDescription;
import org.gecko.rsa.model.rsa.EndpointDescriptions;
import org.gecko.rsa.model.rsa.Property;
import org.gecko.rsa.model.rsa.RSAFactory;
import org.gecko.rsa.model.rsa.Value;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.Constants;
import org.osgi.framework.FrameworkUtil;
import org.osgi.service.remoteserviceadmin.EndpointEvent;
import org.osgi.service.remoteserviceadmin.EndpointEventListener;

@RunWith(MockitoJUnitRunner.class)
public class ImportingDiscoveryIntegrationTest extends AbstractOSGiTest {
	
	private EndpointDiscovery discovery01;
	private EndpointDiscovery discovery02;

	public ImportingDiscoveryIntegrationTest() {
		super(FrameworkUtil.getBundle(ImportingDiscoveryIntegrationTest.class).getBundleContext());
	}

	@Override
	public void doBefore() {
	}
	
	@Override
	public void doAfter() {
		if (discovery01 != null && discovery01.isRunning()) {
			discovery01.stop();
		}
		if (discovery02 != null && discovery02.isRunning()) {
			discovery02.stop();
		}
	}
	
	@Test
	public void testEndpointListenerRegistration() {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		discovery01.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testImportDescriptionsNoServiceId() throws InterruptedException {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		discovery01.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("test");
		p0101.setValue("1");
		p0101.setValueType("Integer");
		d01.getProperty().add(p0101);
		
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		try {
			discovery01.getImportingEndpointManager().addImported(d01);
			fail("This part should no be reached");
		} catch (Exception e) {
			assertTrue(e instanceof IllegalStateException);
		} finally {
			
		}
		
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testImportDescriptionsNoObjectClass() throws InterruptedException {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		discovery01.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointDescriptions descriptions = RSAFactory.eINSTANCE.createEndpointDescriptions();
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("endpoint.id");
		p0101.setValue("1");
		d01.getProperty().add(p0101);
		descriptions.getEndpointDescription().add(d01);
		
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		try {
			discovery01.getImportingEndpointManager().addImported(d01);
			fail("This part should no be reached");
		} catch (Exception e) {
			assertTrue(e instanceof IllegalStateException);
		} 
		
		Property p0102 = RSAFactory.eINSTANCE.createProperty();
		p0102.setName(Constants.OBJECTCLASS);
		Array a0102 = RSAFactory.eINSTANCE.createArray();
		Value v010201 = RSAFactory.eINSTANCE.createValue();
		a0102.getValue().add(v010201);
		FeatureMapUtil.addText(v010201.getMixed(), Object.class.getName());
		p0102.setArray(a0102);
		d01.getProperty().add(p0102);
		descriptions.getEndpointDescription().add(d01);
		
		try {
			discovery01.getImportingEndpointManager().addImported(d01);
			fail("This part should no be reached");
		} catch (Exception e) {
			assertTrue(e instanceof IllegalStateException);
		} 
		
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testImportDescriptionsNoScope() throws InterruptedException {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		discovery01.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("endpoint.id");
		p0101.setValue("1");
		d01.getProperty().add(p0101);
		Property p0102 = RSAFactory.eINSTANCE.createProperty();
		p0102.setName(Constants.OBJECTCLASS);
		Array a0102 = RSAFactory.eINSTANCE.createArray();
		Value v010201 = RSAFactory.eINSTANCE.createValue();
		a0102.getValue().add(v010201);
		FeatureMapUtil.addText(v010201.getMixed(), Object.class.getName());
		p0102.setArray(a0102);
		d01.getProperty().add(p0102);
		Property p0103 = RSAFactory.eINSTANCE.createProperty();
		p0103.setName("service.imported.configs");
		p0103.setValue("test.cnf");
		d01.getProperty().add(p0103);
		
		AtomicBoolean added = new AtomicBoolean(false);
		CountDownLatch addLatch = new CountDownLatch(1);
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				if (EndpointEvent.ADDED == event.getType()) {
					added.set(true);
					addLatch.countDown();
				}
				
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		discovery01.getImportingEndpointManager().addImported(d01);
		assertFalse(addLatch.await(3, TimeUnit.SECONDS));
		assertFalse(added.get());
		
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testImportDescriptionsWrongScope() throws InterruptedException {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		discovery01.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("endpoint.id");
		p0101.setValue("1");
		d01.getProperty().add(p0101);
		Property p0102 = RSAFactory.eINSTANCE.createProperty();
		p0102.setName(Constants.OBJECTCLASS);
		Array a0102 = RSAFactory.eINSTANCE.createArray();
		Value v010201 = RSAFactory.eINSTANCE.createValue();
		a0102.getValue().add(v010201);
		FeatureMapUtil.addText(v010201.getMixed(), Object.class.getName());
		p0102.setArray(a0102);
		d01.getProperty().add(p0102);
		Property p0103 = RSAFactory.eINSTANCE.createProperty();
		p0103.setName("service.imported.configs");
		p0103.setValue("test.cnf");
		d01.getProperty().add(p0103);
		
		AtomicBoolean added = new AtomicBoolean(false);
		CountDownLatch addLatch = new CountDownLatch(1);
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				if (EndpointEvent.ADDED == event.getType()) {
					added.set(true);
					addLatch.countDown();
				}
				
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(EndpointEventListener.ENDPOINT_LISTENER_SCOPE, "(test=true)");
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		discovery01.getImportingEndpointManager().addImported(d01);
		assertFalse(addLatch.await(3, TimeUnit.SECONDS));
		assertFalse(added.get());
		
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testMultipleDiscoveryAddImportDescriptionsSuccess() throws InterruptedException {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		discovery01.start();
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		discovery02 = new EndpointDiscovery(getBundleContext(), "Second");
		discovery02.start();
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("endpoint.id");
		p0101.setValue("1");
		d01.getProperty().add(p0101);
		Property p0102 = RSAFactory.eINSTANCE.createProperty();
		p0102.setName(Constants.OBJECTCLASS);
		Array a0102 = RSAFactory.eINSTANCE.createArray();
		Value v010201 = RSAFactory.eINSTANCE.createValue();
		a0102.getValue().add(v010201);
		FeatureMapUtil.addText(v010201.getMixed(), Object.class.getName());
		p0102.setArray(a0102);
		d01.getProperty().add(p0102);
		Property p0103 = RSAFactory.eINSTANCE.createProperty();
		p0103.setName("service.imported.configs");
		p0103.setValue("test.cnf");
		d01.getProperty().add(p0103);
//		
		AtomicInteger added = new AtomicInteger();
		CountDownLatch addLatch = new CountDownLatch(2);
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				if (EndpointEvent.ADDED == event.getType()) {
					added.incrementAndGet();
					addLatch.countDown();
				}
				
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(EndpointEventListener.ENDPOINT_LISTENER_SCOPE, "(service.imported.configs=test.cnf)");
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
//		
		assertEquals(3, endpointListenerTracker.getCurrentCreateCount(true));
		
		discovery01.getImportingEndpointManager().addImported(d01);
		assertFalse(addLatch.await(3, TimeUnit.SECONDS));
		assertEquals(1, added.get());
		
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
		discovery02.stop();
		assertEquals(2, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testAddImportDescriptionsBeforeTopology() throws InterruptedException {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		discovery01.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("endpoint.id");
		p0101.setValue("1");
		d01.getProperty().add(p0101);
		Property p0102 = RSAFactory.eINSTANCE.createProperty();
		p0102.setName(Constants.OBJECTCLASS);
		Array a0102 = RSAFactory.eINSTANCE.createArray();
		Value v010201 = RSAFactory.eINSTANCE.createValue();
		a0102.getValue().add(v010201);
		FeatureMapUtil.addText(v010201.getMixed(), Object.class.getName());
		p0102.setArray(a0102);
		d01.getProperty().add(p0102);
		Property p0103 = RSAFactory.eINSTANCE.createProperty();
		p0103.setName("service.imported.configs");
		p0103.setValue("test.cnf");
		d01.getProperty().add(p0103);
		
		discovery01.getImportingEndpointManager().addImported(d01);

		AtomicBoolean added = new AtomicBoolean(false);
		CountDownLatch addLatch = new CountDownLatch(1);
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				if (EndpointEvent.ADDED == event.getType()) {
					added.set(true);
					addLatch.countDown();
				}
				
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(EndpointEventListener.ENDPOINT_LISTENER_SCOPE, "(service.imported.configs=test.cnf)");
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		assertTrue(addLatch.await(3, TimeUnit.SECONDS));
		assertTrue(added.get());
		
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testAddImportDescriptionsSuccess() throws InterruptedException {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		discovery01.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("endpoint.id");
		p0101.setValue("1");
		d01.getProperty().add(p0101);
		Property p0102 = RSAFactory.eINSTANCE.createProperty();
		p0102.setName(Constants.OBJECTCLASS);
		Array a0102 = RSAFactory.eINSTANCE.createArray();
		Value v010201 = RSAFactory.eINSTANCE.createValue();
		a0102.getValue().add(v010201);
		FeatureMapUtil.addText(v010201.getMixed(), Object.class.getName());
		p0102.setArray(a0102);
		d01.getProperty().add(p0102);
		Property p0103 = RSAFactory.eINSTANCE.createProperty();
		p0103.setName("service.imported.configs");
		p0103.setValue("test.cnf");
		d01.getProperty().add(p0103);
		
		AtomicBoolean added = new AtomicBoolean(false);
		CountDownLatch addLatch = new CountDownLatch(1);
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				if (EndpointEvent.ADDED == event.getType()) {
					added.set(true);
					addLatch.countDown();
				}
				
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(EndpointEventListener.ENDPOINT_LISTENER_SCOPE, "(service.imported.configs=test.cnf)");
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		discovery01.getImportingEndpointManager().addImported(d01);
		assertTrue(addLatch.await(3, TimeUnit.SECONDS));
		assertTrue(added.get());
		
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testModifyImportDescriptionsSuccess() throws InterruptedException {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		discovery01.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("endpoint.id");
		p0101.setValue("1");
		d01.getProperty().add(p0101);
		Property p0102 = RSAFactory.eINSTANCE.createProperty();
		p0102.setName(Constants.OBJECTCLASS);
		Array a0102 = RSAFactory.eINSTANCE.createArray();
		Value v010201 = RSAFactory.eINSTANCE.createValue();
		a0102.getValue().add(v010201);
		FeatureMapUtil.addText(v010201.getMixed(), Object.class.getName());
		p0102.setArray(a0102);
		d01.getProperty().add(p0102);
		Property p0103 = RSAFactory.eINSTANCE.createProperty();
		p0103.setName("service.imported.configs");
		p0103.setValue("test.cnf");
		d01.getProperty().add(p0103);
		
		AtomicBoolean modified = new AtomicBoolean(false);
		CountDownLatch modifyLatch = new CountDownLatch(1);
		AtomicBoolean added = new AtomicBoolean(false);
		CountDownLatch addLatch = new CountDownLatch(1);
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				if (EndpointEvent.ADDED == event.getType()) {
					added.set(true);
					addLatch.countDown();
				}
				if (EndpointEvent.MODIFIED == event.getType()) {
					modified.set(true);
					modifyLatch.countDown();
				}
				
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(EndpointEventListener.ENDPOINT_LISTENER_SCOPE, "(service.imported.configs=test.cnf)");
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		// first modify should end in an add
		discovery01.getImportingEndpointManager().modifyImported(d01);
		assertTrue(addLatch.await(3, TimeUnit.SECONDS));
		assertTrue(added.get());
		
		// second modify should end in a modification event
		discovery01.getImportingEndpointManager().modifyImported(d01);
		assertTrue(modifyLatch.await(3, TimeUnit.SECONDS));
		assertTrue(modified.get());
		
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
	@Test
	public void testRemoveImportDescriptionsSuccess() throws InterruptedException {
		ServiceChecker<EndpointEventListener> endpointListenerTracker = createCheckerTrackedForCleanUp(EndpointEventListener.class);
		endpointListenerTracker.start();
		assertEquals(0, endpointListenerTracker.getCurrentCreateCount(false));
		
		discovery01 = new EndpointDiscovery(getBundleContext(), "Test");
		discovery01.start();
		
		assertEquals(1, endpointListenerTracker.getCurrentCreateCount(true));
		
		EndpointDescription d01 = RSAFactory.eINSTANCE.createEndpointDescription();
		Property p0101 = RSAFactory.eINSTANCE.createProperty();
		p0101.setName("endpoint.id");
		p0101.setValue("1");
		d01.getProperty().add(p0101);
		Property p0102 = RSAFactory.eINSTANCE.createProperty();
		p0102.setName(Constants.OBJECTCLASS);
		Array a0102 = RSAFactory.eINSTANCE.createArray();
		Value v010201 = RSAFactory.eINSTANCE.createValue();
		a0102.getValue().add(v010201);
		FeatureMapUtil.addText(v010201.getMixed(), Object.class.getName());
		p0102.setArray(a0102);
		d01.getProperty().add(p0102);
		Property p0103 = RSAFactory.eINSTANCE.createProperty();
		p0103.setName("service.imported.configs");
		p0103.setValue("test.cnf");
		d01.getProperty().add(p0103);
		
		AtomicBoolean removed = new AtomicBoolean(false);
		CountDownLatch removeLatch = new CountDownLatch(1);
		AtomicBoolean added = new AtomicBoolean(false);
		CountDownLatch addLatch = new CountDownLatch(1);
		EndpointEventListener topology = new EndpointEventListener() {
			
			@Override
			public void endpointChanged(EndpointEvent event, String filter) {
				if (EndpointEvent.ADDED == event.getType()) {
					added.set(true);
					addLatch.countDown();
				}
				if (EndpointEvent.REMOVED == event.getType()) {
					removed.set(true);
					removeLatch.countDown();
				}
				
			}
		};
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(EndpointEventListener.ENDPOINT_LISTENER_SCOPE, "(service.imported.configs=test.cnf)");
		registerServiceForCleanup(EndpointEventListener.class, topology, props);
		
		assertEquals(2, endpointListenerTracker.getCurrentCreateCount(true));
		
		// first modify should end in an add
		discovery01.getImportingEndpointManager().addImported(d01);
		assertTrue(addLatch.await(3, TimeUnit.SECONDS));
		assertTrue(added.get());
		
		// second modify should end in a remove event
		discovery01.getImportingEndpointManager().removeImported(d01);
		assertTrue(removeLatch.await(3, TimeUnit.SECONDS));
		assertTrue(removed.get());
		
		discovery01.stop();
		assertEquals(1, endpointListenerTracker.getCurrentRemoveCount(true));
	}
	
}
