/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.gecko.rsa.example.consumer;

import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.gecko.rsa.example.api.EchoService;
import org.gecko.rsa.rsatest.Address;
import org.gecko.rsa.rsatest.ContextType;
import org.gecko.rsa.rsatest.Person;
import org.gecko.rsa.rsatest.RSATestFactory;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.util.pushstream.PushStream;

@Component(immediate=true)
public class EchoConsumer {
    
    EchoService echoService;
    
    ScheduledExecutorService ses = Executors.newSingleThreadScheduledExecutor();

    @Activate
    public void activate() {
    	System.out.println("Activate");
        ses.scheduleWithFixedDelay(()->executeStuff(), 0, 120, TimeUnit.SECONDS);
    }
    
    @Deactivate
    public void deactivate() {
    	System.out.println("De-Activate");
    	ses.shutdown();
    	ses.shutdownNow();
    }
    
    public void executeStuff() {
    	String text = "Good morning";
    	System.out.println(String.format("--> Sending to echo service '%s'", text));
    	long start = System.currentTimeMillis();
    	String result = echoService.echo(text);
    	long time = System.currentTimeMillis() - start;
        System.out.println(String.format("--> Received from echo service '%s' (%d ms)", result, time));
        System.out.println("------------");
        text = "mh";
        System.out.println(String.format("--> Sending to getPersonById for id '%s'", text));
        start = System.currentTimeMillis();
        Person p = echoService.getPersonById(text);
        time = System.currentTimeMillis() - start;
        System.out.println(String.format("--> Received person '%s' (%d ms)", p.toString(), time));
        System.out.println("------------");
        text = "jks4";
        System.out.println(String.format("--> Sending to getAddressById for id '%s'", text));
        start = System.currentTimeMillis();
        List<Address> a = echoService.getAddressById(text);
        time = System.currentTimeMillis() - start;
        System.out.println(String.format("--> Received addresses (%d ms)", time));
        a.forEach((adr)->System.out.println(String.format("--> Received address '%s'", adr.toString())));
        System.out.println("------------");
        
        Address ap = RSATestFactory.eINSTANCE.createAddress();
        ap.setContext(ContextType.PRIVATE);
        ap.setCity("Gera");
        ap.setStreet("Kurt-Keicher-Strasse");
        ap.setNumber("93");
        ap.setZip("07545");
        System.out.println(String.format("--> Sending saveAddress for '%s'", ap.toString()));
        start = System.currentTimeMillis();
        boolean r = echoService.saveAddress(ap);
        time = System.currentTimeMillis() - start;
        System.out.println(String.format("--> Received save result '%s' (%d ms)", r, time));
        System.out.println("------------");
        text = "dim";
        System.out.println(String.format("--> Sending getNamesByPrefix for '%s'", text));
        start = System.currentTimeMillis();
        PushStream<String> stream = echoService.getNamesByPrefix(text);
        time = System.currentTimeMillis() - start;
        System.out.println(String.format("--> Received pushstream for prefix '%s' (%d ms)", text, time));
        stream.forEach(s->System.out.println(String.format("--> Received name '%s'", s)));
    }
    

    @Reference
    public void setEchoService(EchoService echoService) {
        this.echoService = echoService;
    }
    
}
