/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.rsa.provider;

import java.util.Dictionary;
import java.util.Hashtable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Logger;

import org.apache.aries.rsa.spi.DistributionProvider;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.osgi.messaging.MessagingService;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.remoteserviceadmin.EndpointEventListener;
import org.osgi.service.remoteserviceadmin.RemoteConstants;
import org.osgi.util.tracker.ServiceTracker;

/**
 * Activator for the messaging adapter based distribution provider
 * @author Mark Hoffmann
 * @since 07.07.2018
 */
public class Activator implements BundleActivator {
	
	private static final Logger logger = Logger.getLogger(Activator.class.getName());
	private ServiceTracker<MessagingService, MessagingService> messagingTracker;
	private ServiceTracker<ResourceSetFactory, ResourceSetFactory> rsfTracker;
	private ServiceRegistration<DistributionProvider> distributionRegistration = null;
	private CountDownLatch latch = null;
	private AtomicReference<MessagingService> messagingService = new AtomicReference<MessagingService>();
	private AtomicReference<ResourceSetFactory> resourceSetFactory = new AtomicReference<ResourceSetFactory>();
	private BundleContext context;

    /* 
     * (non-Javadoc)
     * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
     */
    @Override
    public void start(BundleContext context) throws Exception {
    	this.context = context;
    	latch = new CountDownLatch(2);
    	Executors.newSingleThreadExecutor().submit(()->{
    		try {
				latch.await();
				registerDistributionProvider(messagingService.get(), resourceSetFactory.get());
			} catch (InterruptedException e) {
				logger.warning("Awaiting MessagingService and ResourceSetFactory was interrupted");
			}
    	});
		messagingTracker = new MessageAdapterTracker(context);
		messagingTracker.open();
		rsfTracker = new RSFTracker(context);
		rsfTracker.open();
    }

    /* 
     * (non-Javadoc)
     * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
     */
    @Override
    public void stop(BundleContext context) throws Exception {
    	messagingTracker.close();
    	rsfTracker.close();
        unregisterDistributionProvider();
    }
    
    /**
     * Registers the distribution provider
     * @param messaging the messaging service to inject
     * @param resourceSetFactory the resource set factory
     */
    private void registerDistributionProvider(MessagingService messaging, ResourceSetFactory resourceSetFactory) {
    	if (distributionRegistration != null) {
    		logger.severe("There is already a registration for this distribution provider. No re-registering. Doing nothing");
    		return;
    	}
    	DistributionProvider provider = new MessagingRSAProvider(messaging, resourceSetFactory);
        Dictionary<String, Object> props = new Hashtable<>();
        props.put(RemoteConstants.REMOTE_INTENTS_SUPPORTED, new String[]{ "osgi.basic", "osgi.async" });
        props.put(RemoteConstants.REMOTE_CONFIGS_SUPPORTED, provider.getSupportedTypes());
        distributionRegistration = context.registerService(DistributionProvider.class, provider, props);
    }
    
    /**
     * Un-registering the distribution provider
     */
    private void unregisterDistributionProvider() {
    	if (distributionRegistration != null) {
    		distributionRegistration.unregister();
    		distributionRegistration = null;
    	}
    }
    
    /**
     * Service tracker that tracks {@link EndpointEventListener}
     * @author Mark Hoffmann
     * @since 02.07.2018
     */
    private final class MessageAdapterTracker extends ServiceTracker<MessagingService, MessagingService> {
    	
    	
    	private MessageAdapterTracker(BundleContext context) {
    		super(context, MessagingService.class, null);
    	}
    	
    	/* 
    	 * (non-Javadoc)
    	 * @see org.osgi.util.tracker.ServiceTracker#addingService(org.osgi.framework.ServiceReference)
    	 */
    	@Override
    	public MessagingService addingService(ServiceReference<MessagingService> reference) {
    		MessagingService messaging = super.addingService(reference);
    		messagingService.set(messaging);
    		latch.countDown();
    		return messaging;
    	}
    	
    	/* 
    	 * (non-Javadoc)
    	 * @see org.osgi.util.tracker.ServiceTracker#removedService(org.osgi.framework.ServiceReference, java.lang.Object)
    	 */
    	@Override
    	public void removedService(ServiceReference<MessagingService> reference, MessagingService endpointListener) {
    		unregisterDistributionProvider();
    		super.removedService(reference, endpointListener);
    	}
    }
    
    /**
     * Service tracker that tracks {@link EndpointEventListener}
     * @author Mark Hoffmann
     * @since 02.07.2018
     */
    private final class RSFTracker extends ServiceTracker<ResourceSetFactory, ResourceSetFactory> {
    	
    	
    	private RSFTracker(BundleContext context) {
    		super(context, ResourceSetFactory.class, null);
    	}
    	
    	/* 
    	 * (non-Javadoc)
    	 * @see org.osgi.util.tracker.ServiceTracker#addingService(org.osgi.framework.ServiceReference)
    	 */
    	@Override
    	public ResourceSetFactory addingService(ServiceReference<ResourceSetFactory> reference) {
    		ResourceSetFactory rsf = super.addingService(reference);
    		resourceSetFactory.set(rsf);
    		latch.countDown();
    		return rsf;
    	}
    	
    	/* 
    	 * (non-Javadoc)
    	 * @see org.osgi.util.tracker.ServiceTracker#removedService(org.osgi.framework.ServiceReference, java.lang.Object)
    	 */
    	@Override
    	public void removedService(ServiceReference<ResourceSetFactory> reference, ResourceSetFactory endpointListener) {
    		unregisterDistributionProvider();
    		super.removedService(reference, endpointListener);
    	}
    }

}
