/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.gecko.rsa.topology.exports;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import org.gecko.rsa.api.ExportPolicy;
import org.gecko.rsa.api.helper.FilterHelper;
import org.gecko.rsa.topology.TopologyTPExecutor;
import org.osgi.framework.Bundle;
import org.osgi.framework.Filter;
import org.osgi.framework.ServiceEvent;
import org.osgi.framework.ServiceListener;
import org.osgi.framework.ServiceReference;
import org.osgi.service.remoteserviceadmin.EndpointDescription;
import org.osgi.service.remoteserviceadmin.EndpointEventListener;
import org.osgi.service.remoteserviceadmin.ExportRegistration;
import org.osgi.service.remoteserviceadmin.RemoteConstants;
import org.osgi.service.remoteserviceadmin.RemoteServiceAdmin;

/**
 * Manages exported endpoints of DOSGi services and notifies EndpointListeners of changes.
 * <ul>
 * <li> Tracks local RemoteServiceAdmin instances by using a ServiceTracker
 * <li> Uses a ServiceListener to track local OSGi services
 * <li> When a service is published that is supported by DOSGi the
 *      known RemoteServiceAdmins are instructed to export the service and
 *      the EndpointListeners are notified
 * <li> When a service is unpublished the EndpointListeners are notified.
 *      The endpoints are not closed as the ExportRegistration takes care of this
 * </ul>
 */
public class TopologyManagerExport implements ServiceListener {
	
    private static final Logger logger = Logger.getLogger(TopologyManagerExport.class.getName());

    private final ThreadPoolExecutor executor = new TopologyTPExecutor(5, 10, 50, TimeUnit.SECONDS, "RSA-TM-Export");
    private final Map<RemoteServiceAdmin, ServiceExportsRepository> endpointRepo = new HashMap<>();
    private final Set<ServiceReference<?>> toBeExported = new HashSet<>();
    private ExportPolicy policy;
    private Map<Integer, String> typeNames;
    private final EndpointListenerNotifier notifier;

    public TopologyManagerExport(ExportPolicy policy) {
        this.notifier = new EndpointListenerNotifier();
        this.policy = policy;
        createTypeNames();
    }
    
    /**
     * Called to shutdown and cleanup all resources 
     */
    public void dispose() {
    	notifier.dispose();
    	executor.shutdown();
    }

    private void createTypeNames() {
        this.typeNames = new HashMap<>();
        this.typeNames.put(ServiceEvent.MODIFIED, "modified");
        this.typeNames.put(ServiceEvent.MODIFIED_ENDMATCH, "modified endmatch");
        this.typeNames.put(ServiceEvent.REGISTERED, "registered");
        this.typeNames.put(ServiceEvent.UNREGISTERING, "unregistering");
    }

    // track all service registrations so we can export any services that are configured to be exported
    // ServiceListener events may be delivered out of order, concurrently, re-entrant, etc. (see spec or docs)
    public void serviceChanged(ServiceEvent event) {
        ServiceReference<?> sref = event.getServiceReference();
        if (!shouldExport(sref)) {
            logger.fine(String.format("Skipping service %s", sref));
            return;
        }
        logger.info(String.format("Received ServiceEvent type: %s, sref: %s", getTypeName(event), sref));
        switch (event.getType()) {
        case ServiceEvent.REGISTERED:
            doExport(sref);
            break;
        case ServiceEvent.MODIFIED:
            modified(sref);
            break;
        case ServiceEvent.MODIFIED_ENDMATCH:
            remove(sref);
            break;
        case ServiceEvent.UNREGISTERING:
            remove(sref);
            break;
        }
    }

    private void modified(ServiceReference<?> sref) {
        for (RemoteServiceAdmin rsa : endpointRepo.keySet()) {
            ServiceExportsRepository repo = endpointRepo.get(rsa);
            repo.modifyService(sref);
        }
    }

    private void remove(ServiceReference<?> sref) {
        toBeExported.remove(sref);
        for (RemoteServiceAdmin rsa : endpointRepo.keySet()) {
            ServiceExportsRepository repo = endpointRepo.get(rsa);
            repo.removeService(sref);
        }
    }
    
    public String getTypeName(ServiceEvent event) {
        return typeNames.get(event.getType());
    }

    public void add(RemoteServiceAdmin rsa) {
        endpointRepo.put(rsa,  new ServiceExportsRepository(rsa, notifier));
        for (ServiceReference<?> serviceRef : toBeExported) {
            exportInBackground(serviceRef);
        }
    };

    public void remove(RemoteServiceAdmin rsa) {
        ServiceExportsRepository repo = endpointRepo.remove(rsa);
        if (repo != null) {
            repo.close();
        }
    };

    private void exportInBackground(final ServiceReference<?> sref) {
        executor.execute(new Runnable() {
            public void run() {
                doExport(sref);
            }
        });
    }

    private void doExport(final ServiceReference<?> sref) {
        logger.fine(String.format("Exporting service %s", sref));
        toBeExported.add(sref);
        if (endpointRepo.size() == 0) {
            logger.severe(String.format("Unable to export service from bundle %s, interfaces: %s as no RemoteServiceAdmin is available. Marked for later export.",
                    getSymbolicName(sref.getBundle()),
                    sref.getProperty(org.osgi.framework.Constants.OBJECTCLASS)));
            return;
        }

        for (RemoteServiceAdmin remoteServiceAdmin : endpointRepo.keySet()) {
            ServiceExportsRepository repo = endpointRepo.get(remoteServiceAdmin);
            Collection<ExportRegistration> regs = exportService(remoteServiceAdmin, sref);
            repo.addService(sref, regs);
        }
    }
    
    private Collection<ExportRegistration> exportService(
            final RemoteServiceAdmin rsa,
            final ServiceReference<?> sref) {
        // abort if the service was unregistered by the time we got here
        // (we check again at the end, but this optimization saves unnecessary heavy processing)
        if (sref.getBundle() == null) {
            logger.info(String.format("TopologyManager: export aborted for %s since it was unregistered", sref));
            return Collections.emptyList();
        }

        logger.fine(String.format("Exporting Service %s using RemoteServiceAdmin %s", sref, rsa.getClass().getName()));
        Map<String, ?> addProps = policy.additionalParameters(sref);
        Collection<ExportRegistration> exportRegs = rsa.exportService(sref, addProps);

        // process successful/failed registrations
        for (ExportRegistration reg : exportRegs) {
            if (reg.getException() == null) {
                EndpointDescription endpoint = reg.getExportReference().getExportedEndpoint();
                logger.info(String.format("TopologyManager: export succeeded for %s, endpoint %s, rsa %s", sref, endpoint, rsa.getClass().getName()));
            } else {
                logger.severe(String.format("TopologyManager: export failed for %s", sref, reg.getException()));
                reg.close();
            }
        }

        // abort export if service was unregistered in the meanwhile (since we have a race
        // with the unregister event which may have already been handled, so we'll miss it)
        if (sref.getBundle() == null) {
            logger.info(String.format("TopologyManager: export reverted for %s since service was unregistered", sref));
            for (ExportRegistration reg : exportRegs) {
                reg.close();
            }
        }
        
        return exportRegs;
    }

    private boolean shouldExport(ServiceReference<?> sref) {
        Map<String, ?> addProps = policy.additionalParameters(sref);
        List<String> exported= FilterHelper.normalize(sref.getProperty(RemoteConstants.SERVICE_EXPORTED_INTERFACES));
        List<String> addExported = FilterHelper.normalize(addProps.get(RemoteConstants.SERVICE_EXPORTED_INTERFACES));
        return sizeOf(exported) + sizeOf(addExported) > 0;
    }

    private int sizeOf(List<String> list) {
        return list == null ? 0 : list.size();
    }

    private Object getSymbolicName(Bundle bundle) {
        return bundle == null ? null : bundle.getSymbolicName();
    }

    public void addEPListener(EndpointEventListener epListener, Set<Filter> filters) {
        Collection<EndpointDescription> endpoints = new ArrayList<>();
        for (RemoteServiceAdmin rsa : endpointRepo.keySet()) {
            ServiceExportsRepository repo = endpointRepo.get(rsa);
            endpoints.addAll(repo.getAllEndpoints());
        }
        notifier.add(epListener, filters, endpoints);
    }

    public void removeEPListener(EndpointEventListener listener) {
        notifier.remove(listener);
    }
}
