/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.gecko.rsa.topology;

import java.util.ArrayList;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;
import java.util.logging.Logger;

import org.gecko.rsa.core.helper.ReferenceCounter;
import org.gecko.rsa.topology.exports.TopologyManagerExport;
import org.gecko.rsa.topology.imports.TopologyManagerImport;
import org.gecko.rsa.topology.imports.local.LocalServiceListenerHook;
import org.gecko.rsa.topology.imports.local.ServiceInterestListener;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.framework.hooks.service.ListenerHook;
import org.osgi.service.remoteserviceadmin.EndpointEvent;
import org.osgi.service.remoteserviceadmin.EndpointEventListener;

/**
 * Manages the end-point listener for the import of external services.
 * The end-point listener scope reflects the combined filters of all services 
 * that are asked for (by listeners and service lookups) in the current system. 
 * 
 * Discovery will then send callbacks when external end-points are added / removed that match
 * the interest in the local system.
 */
public class EndpointListenerManager implements ServiceInterestListener{

    private final class EndpointListenerAdapter implements EndpointEventListener {

        /* 
         * (non-Javadoc)
         * @see org.osgi.service.remoteserviceadmin.EndpointEventListener#endpointChanged(org.osgi.service.remoteserviceadmin.EndpointEvent, java.lang.String)
         */
        @Override
        public void endpointChanged(EndpointEvent event, String filter) {
            tmImport.endpointChanged(event, filter);
        }
    }
    
    private static final Logger logger = Logger.getLogger(EndpointListenerManager.class.getName());

    private final BundleContext bctx;
    private volatile ServiceRegistration<EndpointEventListener> eelRegistration;
    private volatile ServiceRegistration<ListenerHook> serviceHookRegistration;
    
    private final List<String> filters = new ArrayList<String>();
    private final TopologyManagerImport tmImport;
    private final TopologyManagerExport tmExport;
    private final LocalServiceListenerHook listenerHook;
    private final EndpointEventListenerTracker exportTracker;
    
    /**
     * Count service interest by filter. This allows to modify the scope of the EndpointListener as seldom as possible
     */
    private final ReferenceCounter<String> importInterestsCounter = new ReferenceCounter<String>();




    public EndpointListenerManager(BundleContext bc, TopologyManagerImport tmImport, TopologyManagerExport tmExport) {
        this.bctx = bc;
        this.tmImport = tmImport;
		this.tmExport = tmExport;
        this.listenerHook = new LocalServiceListenerHook(bc, this);
        this.exportTracker = new EndpointEventListenerTracker(bc, tmExport);
    }

    /**
     * Start the end-point listener manager
     */
    public void start() {
    	exportTracker.open();
    	EndpointEventListener endpointListenerAdapter = new EndpointListenerAdapter();
    	// register the delegate to the given end-point event listener
    	eelRegistration = bctx.registerService(EndpointEventListener.class, endpointListenerAdapter,getEELProperties());
        serviceHookRegistration = bctx.registerService(ListenerHook.class, listenerHook, null);
        bctx.addServiceListener(tmExport);
        tmImport.start();
    }

    /**
     * Stops the end-point listener manager
     */
    public void stop() {
    	exportTracker.close();
    	bctx.removeServiceListener(tmExport);
    	tmImport.dispose();
    	tmExport.dispose();
    	if (eelRegistration != null) {
    		eelRegistration.unregister();
        }
        if (serviceHookRegistration != null) {
        	serviceHookRegistration.unregister();
        }
    }

    /* 
	 * (non-Javadoc)
	 * @see org.gecko.rsa.topology.imports.local.ServiceInterestListener#addServiceInterest(java.lang.String)
	 */
	@Override
	public void addServiceInterest(String filter) {
	    if (importInterestsCounter.add(filter) == 1) {
	        extendScope(filter);
	    }
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.rsa.topology.imports.local.ServiceInterestListener#removeServiceInterest(java.lang.String)
	 */
	@Override
	public void removeServiceInterest(String filter) {
	    if (importInterestsCounter.remove(filter) == 0) {
	        logger.fine(String.format("Last reference of import interests is gone. Removing interest filter: %s", filter));
	        reduceScope(filter);
	    }
	}

	/**
	 * Create a copy of the current filter list
	 * @return a copy of the current filter list
	 */
	public List<String> copyFilters() {
	    synchronized (filters) {
	        return new ArrayList<>(filters);
	    }
	}

	/**
	 * Extend the end-point event listener filter scope by the give nfilter 
	 * @param filter the filter to add to the scope
	 */
	protected void extendScope(String filter) {
        if (filter == null) {
            return;
        }
        logger.fine(String.format("EndpointEventListener: Extending scope by %s", filter));
        synchronized (filters) {
            if (filters.add(filter)) {
            	updateRegistration();
            }
        }
    }

    /**
     * Reduces the current end-point event listener scope by the given filter
     * @param filter the filter ro be removed from the scope
     */
    protected void reduceScope(String filter) {
        if (filter == null) {
            return;
        }
        logger.fine(String.format("EndpointEventListener: Reducing scope by %s", filter));
        synchronized (filters) {
            if(filters.remove(filter)) {
            	updateRegistration();
            }
        }
    }
    
    /**
     * Updates the service registration for the end-point event listener
     */
    private void updateRegistration() {
        if (eelRegistration != null) {
        	eelRegistration.setProperties(getEELProperties());
        }
    }
    
    /**
     * Returns the end-point event listener properties 
     * @return the end-point event listener properties 
     */
    private Dictionary<String, Object> getEELProperties() {
        Dictionary<String, Object> p = new Hashtable<String, Object>();
        p.put(EndpointEventListener.ENDPOINT_LISTENER_SCOPE, copyFilters());
        return p;
    }

}
