/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.gecko.rsa.topology.exports;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;

import org.gecko.rsa.core.helper.EndpointHelper;
import org.osgi.framework.Filter;
import org.osgi.service.remoteserviceadmin.EndpointDescription;
import org.osgi.service.remoteserviceadmin.EndpointEvent;
import org.osgi.service.remoteserviceadmin.EndpointEventListener;

/**
 * Tracks {@link EndpointEventListener}'s to be added and removed and allows to notify them of end-points.
 */
public class EndpointListenerNotifier {
	
    private static final Logger logger = Logger.getLogger(EndpointListenerNotifier.class.getName());
    private final Map<EndpointEventListener, Set<Filter>> listeners = new ConcurrentHashMap<EndpointEventListener, Set<Filter>>();

    public EndpointListenerNotifier() {
    }
   
    /**
     * Adds an {@link EndpointEventListener} with a set of filters and a collection of endpoint descriptions
     * @param eepListener the {@link EndpointEventListener}
     * @param filters all filters
     * @param endpoints all end-point descriptions
     */
    public void add(EndpointEventListener eepListener, Set<Filter> filters, Collection<EndpointDescription> endpoints) {
        logger.fine("EndpointEventListener added");
        listeners.put(eepListener, filters);
        endpoints.stream().
        	map(ep->new EndpointEvent(EndpointEvent.ADDED, ep)).
        	forEach(ev->notifyListener(ev, eepListener, filters));
    }
    
    /**
     * Removes an end-point listener 
     * @param listener the listener to remove
     */
    public void remove(EndpointEventListener listener) {
        logger.fine("EndpointEventListener removed");
        listeners.remove(listener);
    }
    
    /**
     * Sends the given event to all available listeners
     * @param event the event to be send
     */
    public void sendEvent(EndpointEvent event) {
    	listeners.forEach((l,f)->notifyListener(event, l, f));
    }
    
    /**
     * Cleans up all resources
     */
    public void dispose() {
    	listeners.clear();
    }

    /**
     * Notifies the listener for every filter that matches the end-point from the event, that the 
     * end-point has changed
     * @param event the end-point event
     * @param listener the listener to notify
     * @param filters all filters, to get matching filters from
     */
    private void notifyListener(EndpointEvent event, EndpointEventListener listener, Set<Filter> filters) {
        Set<Filter> matchingFilters = EndpointHelper.getMatchingFilters(filters, event.getEndpoint());
        matchingFilters.forEach(f->listener.endpointChanged(event, f.toString()));
    }
    
}
